import {
  Controller,
  Get,
  Post,
  Patch,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  Request,
  Logger,
  BadRequestException,
  UseInterceptors,
  UploadedFile,
} from '@nestjs/common';
import { UsersService } from './users.service';
import { CreateUserDto } from './dto/create-user.dto';
import { UpdateUserDto } from './dto/update-user.dto';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { PermissionsGuard } from '../auth/guards/permissions.guard';
import { Roles } from '../auth/decorators/roles.decorator';
import { RequirePermission } from '../auth/decorators/require-permission.decorator';
import { UserRole } from '../roles/roles.model';
import { FileInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { User } from './entities/user.entity';

interface RequestWithUser extends Request {
  user: User;
  requestingUser?: User;
}

@Controller('api/users')
@UseGuards(JwtAuthGuard, PermissionsGuard)
export class UsersController {
  private readonly logger = new Logger(UsersController.name);

  constructor(private readonly usersService: UsersService) {}

  @Post('employees')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can add employees')
  async create(@Body() createUserDto: CreateUserDto, @Request() req: RequestWithUser) {
    this.logger.debug(`Creating new user with DTO: ${JSON.stringify(createUserDto)}`);
    return this.usersService.create(createUserDto, req.user.id);
  }

  @Get('employees')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can view employees')
  async findAll(
    @Query('limit') limit: string,
    @Query('offset') offset: string,
    @Query('search') search: string,
    @Query('isActive') isActive: string,
    @Request() req: RequestWithUser,
  ) {
    const parsedLimit = limit ? parseInt(limit, 10) : 10;
    const parsedOffset = offset ? parseInt(offset, 10) : 0;
    const parsedIsActive = isActive === 'true' ? true : isActive === 'false' ? false : undefined;

    if (isNaN(parsedLimit) || isNaN(parsedOffset)) {
      throw new BadRequestException('Limit and offset must be valid numbers');
    }

    this.logger.debug(
      `Fetching users with limit=${parsedLimit}, offset=${parsedOffset}, search=${search}, isActive=${parsedIsActive}`,
    );

    return this.usersService.findAll(parsedLimit, parsedOffset, search, parsedIsActive, req.requestingUser);
  }

  @Get('employees/deactivatedlist')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can view employees')
  async findDeactivatedUsers(@Request() req: RequestWithUser) {
    this.logger.debug('Fetching deactivated users');
    return this.usersService.findDeactivatedUsers(req.user, (req as any).ip);
  }

  @Get('employees/:id')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can view employees')
  async findOne(@Param('id') id: string, @Request() req: RequestWithUser) {
    this.logger.debug(`Fetching user with ID ${id}`);
    const parsedId = parseInt(id, 10);
    if (isNaN(parsedId)) {
      throw new BadRequestException('Invalid user ID');
    }
    return this.usersService.findOne(parsedId, req.requestingUser);
  }

  @Patch('employees/:id')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can change employees')
  async update(@Param('id') id: string, @Body() updateUserDto: UpdateUserDto, @Request() req: Request) {
    this.logger.debug(`Updating user with ID ${id} with DTO: ${JSON.stringify(updateUserDto)}`);
    const parsedId = parseInt(id, 10);
    if (isNaN(parsedId)) {
      throw new BadRequestException('Invalid user ID');
    }
    return this.usersService.update(parsedId, updateUserDto);
  }

  @Patch('employees/:id/reactivate')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can change employees')
  async reactivate(@Param('id') id: string, @Request() req: Request) {
    this.logger.debug(`Reactivating user with ID ${id}`);
    const parsedId = parseInt(id, 10);
    if (isNaN(parsedId)) {
      throw new BadRequestException('Invalid user ID');
    }
    return this.usersService.reactivate(parsedId);
  }

  @Patch('employees/:id/deactivate')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can change employees')
  async deactivate(@Param('id') id: string, @Request() req: Request) {
    this.logger.debug(`Deactivating user with ID ${id}`);
    const parsedId = parseInt(id, 10);
    if (isNaN(parsedId)) {
      throw new BadRequestException('Invalid user ID');
    }
    return this.usersService.deactivate(parsedId);
  }

  @Delete('employees/:id')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can delete employees')
  async remove(@Param('id') id: string, @Request() req: Request) {
    this.logger.debug(`Deleting user with ID ${id}`);
    const parsedId = parseInt(id, 10);
    if (isNaN(parsedId)) {
      throw new BadRequestException('Invalid user ID');
    }
    return this.usersService.remove(parsedId);
  }

  @Post('employees/bulk')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can add employees')
  @UseInterceptors(
    FileInterceptor('file', {
      storage: diskStorage({
        destination: './uploads',
      }),
      limits: {
        fileSize: 5 * 1024 * 1024, // 5MB limit
      },
      fileFilter: (req, file, callback) => {
        if (!file.originalname.match(/\.(xlsx|xls)$/)) {
          return callback(new BadRequestException('Only Excel files are allowed!'), false);
        }
        callback(null, true);
      },
    }),
  )
  async bulkCreate(@UploadedFile() file: Express.Multer.File, @Request() req: Request) {
    this.logger.debug(`Bulk creating users from file: ${file?.originalname}`);
    if (!file) {
      throw new BadRequestException('File is required');
    }
    return this.usersService.bulkCreate(file);
  }

  // --- ASSIGNEMPLOYEES ROUTES (duplicated from employees) ---
  @Post('assignemployees')
  @RequirePermission('Can change Assigned To')
  async createAssign(@Body() createUserDto: CreateUserDto, @Request() req: RequestWithUser) {
    this.logger.debug(`Creating new user (assignemployees) with DTO: ${JSON.stringify(createUserDto)}`);
    return this.usersService.create(createUserDto, req.user.id);
  }

  @Get('assignemployees')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can view employees')
  async findAllAssign(
    @Query('limit') limit: string,
    @Query('offset') offset: string,
    @Query('search') search: string,
    @Query('isActive') isActive: string,
    @Request() req: RequestWithUser,
  ) {
    const parsedLimit = limit ? parseInt(limit, 10) : 10;
    const parsedOffset = offset ? parseInt(offset, 10) : 0;
    const parsedIsActive = isActive === 'true' ? true : isActive === 'false' ? false : undefined;

    if (isNaN(parsedLimit) || isNaN(parsedOffset)) {
      throw new BadRequestException('Limit and offset must be valid numbers');
    }

    this.logger.debug(
      `Fetching users (assignemployees) with limit=${parsedLimit}, offset=${parsedOffset}, search=${search}, isActive=${parsedIsActive}`,
    );

    return this.usersService.findAll(parsedLimit, parsedOffset, search, parsedIsActive, req.requestingUser);
  }

  @Get('assignemployees/deactivatedlist')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can view employees')
  async findDeactivatedAssign(@Request() req: RequestWithUser) {
    this.logger.debug('Fetching deactivated users (assignemployees)');
    return this.usersService.findDeactivatedUsers(req.user, (req as any).ip);
  }

  @Get('assignemployees/:id')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can view employees')
  async findOneAssign(@Param('id') id: string, @Request() req: RequestWithUser) {
    this.logger.debug(`Fetching user (assignemployees) with ID ${id}`);
    const parsedId = parseInt(id, 10);
    if (isNaN(parsedId)) {
      throw new BadRequestException('Invalid user ID');
    }
    return this.usersService.findOne(parsedId, req.requestingUser);
  }

  @Patch('assignemployees/:id')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can change employees')
  async updateAssign(@Param('id') id: string, @Body() updateUserDto: UpdateUserDto, @Request() req: Request) {
    this.logger.debug(`Updating user (assignemployees) with ID ${id} with DTO: ${JSON.stringify(updateUserDto)}`);
    const parsedId = parseInt(id, 10);
    if (isNaN(parsedId)) {
      throw new BadRequestException('Invalid user ID');
    }
    return this.usersService.update(parsedId, updateUserDto);
  }

  @Patch('assignemployees/:id/reactivate')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can change employees')
  async reactivateAssign(@Param('id') id: string, @Request() req: Request) {
    this.logger.debug(`Reactivating user (assignemployees) with ID ${id}`);
    const parsedId = parseInt(id, 10);
    if (isNaN(parsedId)) {
      throw new BadRequestException('Invalid user ID');
    }
    return this.usersService.reactivate(parsedId);
  }

  @Patch('assignemployees/:id/deactivate')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can change employees')
  async deactivateAssign(@Param('id') id: string, @Request() req: Request) {
    this.logger.debug(`Deactivating user (assignemployees) with ID ${id}`);
    const parsedId = parseInt(id, 10);
    if (isNaN(parsedId)) {
      throw new BadRequestException('Invalid user ID');
    }
    return this.usersService.deactivate(parsedId);
  }

  @Delete('assignemployees/:id')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can delete employees')
  async removeAssign(@Param('id') id: string, @Request() req: Request) {
    this.logger.debug(`Deleting user (assignemployees) with ID ${id}`);
    const parsedId = parseInt(id, 10);
    if (isNaN(parsedId)) {
      throw new BadRequestException('Invalid user ID');
    }
    return this.usersService.remove(parsedId);
  }

  @Post('assignemployees/bulk')
  @Roles(UserRole.ADMIN, UserRole.OWNER)
  @RequirePermission('Can add employees')
  @UseInterceptors(
    FileInterceptor('file', {
      storage: diskStorage({
        destination: './uploads',
      }),
      limits: {
        fileSize: 5 * 1024 * 1024, // 5MB limit
      },
      fileFilter: (req, file, callback) => {
        if (!file.originalname.match(/\.(xlsx|xls)$/)) {
          return callback(new BadRequestException('Only Excel files are allowed!'), false);
        }
        callback(null, true);
      },
    }),
  )
  async bulkCreateAssign(@UploadedFile() file: Express.Multer.File, @Request() req: Request) {
    this.logger.debug(`Bulk creating users (assignemployees) from file: ${file?.originalname}`);
    if (!file) {
      throw new BadRequestException('File is required');
    }
    return this.usersService.bulkCreate(file);
  }
}