import { Injectable, NotFoundException, Logger } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Role } from './entities/role.entity';
import { PermissionsService } from '../permissions/permissions.service';
import { RolesResponse, RoleWithPermissions } from './roles.model';

@Injectable()
export class RolesService {
  private readonly logger = new Logger(RolesService.name);

  constructor(
    @InjectRepository(Role)
    private readonly roleRepository: Repository<Role>,
    private readonly permissionsService: PermissionsService,
  ) {}

  async findAll(): Promise<RolesResponse> {
    try {
      this.logger.debug('findAll() - Starting to fetch all roles');
      
      const roles = await this.roleRepository.find();
      this.logger.debug(`findAll() - Found ${roles.length} roles in database`);
      
      if (roles.length === 0) {
        this.logger.warn('findAll() - No roles found in database');
        return {
          data: [],
          success: true,
          message: 'List Successfully Retrieved!',
          status: 200
        };
      }
      
      this.logger.debug('findAll() - Fetching permissions for each role');
      const rolesWithPermissions = await Promise.all(
        roles.map(async (role) => {
          try {
            this.logger.debug(`findAll() - Fetching permissions for role ID: ${role.id}, name: ${role.name}`);
            const permissionsResponse = await this.permissionsService.findAll(role.id.toString());
            this.logger.debug(`findAll() - Found ${permissionsResponse.data?.length || 0} permissions for role ID: ${role.id}`);
            return {
              id: role.id,
              name: role.name,
              permissions: permissionsResponse.data
            };
          } catch (error) {
            this.logger.error(`findAll() - Error fetching permissions for role ID ${role.id}: ${error.message}`, error.stack);
            // Continue with empty permissions if there's an error fetching them
            return {
              id: role.id,
              name: role.name,
              permissions: []
            };
          }
        })
      );

      this.logger.log(`findAll() - Successfully retrieved ${rolesWithPermissions.length} roles with permissions`);
      return {
        data: rolesWithPermissions,
        success: true,
        message: 'List Successfully Retrieved!',
        status: 200
      };
    } catch (error) {
      this.logger.error(`findAll() - Error fetching roles: ${error.message}`, error.stack);
      throw error;
    }
  }

  async findOne(id: number): Promise<Role> {
    try {
      this.logger.debug(`findOne() - Searching for role with ID: ${id}`);
      
      const role = await this.roleRepository.findOne({ where: { id } });
      
      if (!role) {
        this.logger.warn(`findOne() - Role with ID ${id} not found`);
        throw new NotFoundException(`Role with ID ${id} not found`);
      }
      
      this.logger.log(`findOne() - Successfully found role with ID: ${id}, name: ${role.name}`);
      return role;
    } catch (error) {
      if (error instanceof NotFoundException) {
        throw error;
      }
      this.logger.error(`findOne() - Error finding role with ID ${id}: ${error.message}`, error.stack);
      throw error;
    }
  }
} 