import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  Query,
  HttpStatus,
  UseGuards,
  Logger,
} from '@nestjs/common';
import { ProductsService } from './products.service';
import { CreateProductDto } from './dto/create-product.dto';
import { UpdateProductDto } from './dto/update-product.dto';
import { ListProductsDto } from './dto/list-products.dto';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { PermissionsGuard } from '../auth/guards/permissions.guard';
import { RolesGuard } from '../auth/guards/roles.guard';

@Controller('api/products')
@UseGuards(JwtAuthGuard, PermissionsGuard, RolesGuard)
export class ProductsController {
  private readonly logger = new Logger(ProductsController.name);

  constructor(private readonly productsService: ProductsService) {}

  @Post()
  async create(@Body() createProductDto: CreateProductDto) {
    try {
      this.logger.debug(`POST /api/products - Creating product with name: ${createProductDto.name}`);
      const product = await this.productsService.create(createProductDto);
      this.logger.log(`POST /api/products - Successfully created product with ID: ${product.id}, name: ${product.name}`);
      return {
        statusCode: HttpStatus.CREATED,
        message: 'Product created successfully',
        data: product,
      };
    } catch (error) {
      const errorType = error.constructor.name;
      const errorMessage = error.message || 'Unknown error';
      this.logger.error(`POST /api/products - Error creating product. Error Type: ${errorType}, Message: ${errorMessage}`, error.stack);
      throw error;
    }
  }

  @Get()
  async findAll(@Query() listProductsDto: ListProductsDto) {
    try {
      this.logger.debug(`GET /api/products - Finding products with filters: ${JSON.stringify(listProductsDto)}`);
      const { data, total } = await this.productsService.findAll(listProductsDto);
      this.logger.log(`GET /api/products - Successfully retrieved ${data.length} products (total: ${total})`);
      return {
        statusCode: HttpStatus.OK,
        message: 'Products retrieved successfully',
        data,
        total,
      };
    } catch (error) {
      const errorType = error.constructor.name;
      const errorMessage = error.message || 'Unknown error';
      this.logger.error(`GET /api/products - Error finding products. Error Type: ${errorType}, Message: ${errorMessage}`, error.stack);
      throw error;
    }
  }

  @Get(':id')
  async findOne(@Param('id') id: string) {
    try {
      const productId = +id;
      this.logger.debug(`GET /api/products/${id} - Finding product with ID: ${productId}`);
      const product = await this.productsService.findOne(productId);
      this.logger.log(`GET /api/products/${id} - Successfully retrieved product: ${product.name}`);
      return {
        statusCode: HttpStatus.OK,
        message: 'Product retrieved successfully',
        data: product,
      };
    } catch (error) {
      const errorType = error.constructor.name;
      const errorMessage = error.message || 'Unknown error';
      this.logger.error(`GET /api/products/${id} - Error finding product. Error Type: ${errorType}, Message: ${errorMessage}`, error.stack);
      throw error;
    }
  }

  @Patch(':id')
  async update(@Param('id') id: string, @Body() updateProductDto: UpdateProductDto) {
    try {
      const productId = +id;
      this.logger.debug(`PATCH /api/products/${id} - Updating product with ID: ${productId}, data: ${JSON.stringify(updateProductDto)}`);
      const product = await this.productsService.update(productId, updateProductDto);
      this.logger.log(`PATCH /api/products/${id} - Successfully updated product: ${product.name}`);
      return {
        statusCode: HttpStatus.OK,
        message: 'Product updated successfully',
        data: product,
      };
    } catch (error) {
      const errorType = error.constructor.name;
      const errorMessage = error.message || 'Unknown error';
      this.logger.error(`PATCH /api/products/${id} - Error updating product. Error Type: ${errorType}, Message: ${errorMessage}`, error.stack);
      throw error;
    }
  }

  @Delete(':id')
  async remove(@Param('id') id: string) {
    try {
      const productId = +id;
      this.logger.debug(`DELETE /api/products/${id} - Deleting product with ID: ${productId}`);
      await this.productsService.remove(productId);
      this.logger.log(`DELETE /api/products/${id} - Successfully deleted product`);
      return {
        statusCode: HttpStatus.OK,
        message: 'Product deleted successfully',
      };
    } catch (error) {
      const errorType = error.constructor.name;
      const errorMessage = error.message || 'Unknown error';
      this.logger.error(`DELETE /api/products/${id} - Error deleting product. Error Type: ${errorType}, Message: ${errorMessage}`, error.stack);
      throw error;
    }
  }
}


