import { Injectable, NotFoundException, ConflictException, BadRequestException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { PostageServiceCharge } from './entities/postage-service-charge.entity';
import { CreatePostageServiceChargeDto } from './dto/create-postage-service-charge.dto';
import { UpdatePostageServiceChargeDto } from './dto/update-postage-service-charge.dto';

@Injectable()
export class PostageServiceChargesService {
  constructor(
    @InjectRepository(PostageServiceCharge)
    private readonly postageServiceChargesRepository: Repository<PostageServiceCharge>,
  ) {}

  async create(createDto: CreatePostageServiceChargeDto): Promise<PostageServiceCharge> {
    try {
      // Check if charge with same name already exists
      const existing = await this.postageServiceChargesRepository.findOne({
        where: { name: createDto.name },
      });

      if (existing) {
        throw new ConflictException(`Postage/Service charge with name "${createDto.name}" already exists`);
      }

      const charge = this.postageServiceChargesRepository.create(createDto);
      return await this.postageServiceChargesRepository.save(charge);
    } catch (error) {
      if (error instanceof ConflictException) {
        throw error;
      }
      throw new BadRequestException(`Failed to create postage/service charge: ${error.message}`);
    }
  }

  async findAll(): Promise<PostageServiceCharge[]> {
    return await this.postageServiceChargesRepository.find({
      where: { isActive: true },
      order: { createdAt: 'DESC' },
    });
  }

  async findAllIncludingInactive(): Promise<PostageServiceCharge[]> {
    return await this.postageServiceChargesRepository.find({
      order: { createdAt: 'DESC' },
    });
  }

  async findOne(id: number): Promise<PostageServiceCharge> {
    const charge = await this.postageServiceChargesRepository.findOne({
      where: { id },
    });

    if (!charge) {
      throw new NotFoundException(`Postage/Service charge with ID ${id} not found`);
    }

    return charge;
  }

  async update(id: number, updateDto: UpdatePostageServiceChargeDto): Promise<PostageServiceCharge> {
    const charge = await this.findOne(id);

    // Check if name is being updated and if it conflicts with existing charge
    if (updateDto.name && updateDto.name !== charge.name) {
      const existing = await this.postageServiceChargesRepository.findOne({
        where: { name: updateDto.name },
      });

      if (existing) {
        throw new ConflictException(`Postage/Service charge with name "${updateDto.name}" already exists`);
      }
    }

    Object.assign(charge, updateDto);
    return await this.postageServiceChargesRepository.save(charge);
  }

  async remove(id: number): Promise<void> {
    const charge = await this.findOne(id);
    // Soft delete
    charge.isActive = false;
    await this.postageServiceChargesRepository.save(charge);
  }
}

