import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  Query,
  HttpStatus,
  UseGuards,
  Logger,
} from '@nestjs/common';
import { OrdersService } from './orders.service';
import { CreateOrderDto } from './dto/create-order.dto';
import { UpdateOrderDto } from './dto/update-order.dto';
import { ListOrdersDto } from './dto/list-orders.dto';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { PermissionsGuard } from '../auth/guards/permissions.guard';
import { RolesGuard } from '../auth/guards/roles.guard';
import { User } from '../auth/decorators/user.decorator';

@Controller('api/orders')
@UseGuards(JwtAuthGuard, PermissionsGuard, RolesGuard)
export class OrdersController {
  private readonly logger = new Logger(OrdersController.name);

  constructor(private readonly ordersService: OrdersService) {}

  @Post()
  async create(@Body() createOrderDto: CreateOrderDto, @User() user: any) {
    try {
      this.logger.debug(`POST /api/orders - Creating order for customer ID: ${createOrderDto.customerId}, user ID: ${user.id}`);
      const order = await this.ordersService.create(createOrderDto, user);
      this.logger.log(`POST /api/orders - Successfully created order with ID: ${order.id}, orderNumber: ${order.orderNumber}`);
      return {
        statusCode: HttpStatus.CREATED,
        message: 'Order created successfully',
        data: order,
      };
    } catch (error) {
      const errorType = error.constructor.name;
      const errorMessage = error.message || 'Unknown error';
      this.logger.error(`POST /api/orders - Error creating order. Error Type: ${errorType}, Message: ${errorMessage}`, error.stack);
      throw error;
    }
  }

  @Get()
  async findAll(@Query() listOrdersDto: ListOrdersDto, @User() user: any) {
    try {
      this.logger.debug(`GET /api/orders - Finding orders with filters: ${JSON.stringify(listOrdersDto)}, user ID: ${user.id}`);
      const { data, total } = await this.ordersService.findAll(listOrdersDto, user);
      this.logger.log(`GET /api/orders - Successfully retrieved ${data.length} orders (total: ${total})`);
      return {
        statusCode: HttpStatus.OK,
        message: 'Orders retrieved successfully',
        data,
        total,
      };
    } catch (error) {
      const errorType = error.constructor.name;
      const errorMessage = error.message || 'Unknown error';
      this.logger.error(`GET /api/orders - Error finding orders. Error Type: ${errorType}, Message: ${errorMessage}`, error.stack);
      throw error;
    }
  }

  @Get(':id')
  async findOne(@Param('id') id: string, @User() user: any) {
    try {
      const orderId = +id;
      this.logger.debug(`GET /api/orders/${id} - Finding order with ID: ${orderId}, user ID: ${user.id}`);
      const order = await this.ordersService.findOne(orderId, user);
      this.logger.log(`GET /api/orders/${id} - Successfully retrieved order: ${order.orderNumber}`);
      return {
        statusCode: HttpStatus.OK,
        message: 'Order retrieved successfully',
        data: order,
      };
    } catch (error) {
      const errorType = error.constructor.name;
      const errorMessage = error.message || 'Unknown error';
      this.logger.error(`GET /api/orders/${id} - Error finding order. Error Type: ${errorType}, Message: ${errorMessage}`, error.stack);
      throw error;
    }
  }

  @Patch(':id')
  async update(
    @Param('id') id: string,
    @Body() updateOrderDto: UpdateOrderDto,
    @User() user: any,
  ) {
    try {
      const orderId = +id;
      this.logger.debug(`PATCH /api/orders/${id} - Updating order with ID: ${orderId}, user ID: ${user.id}, data: ${JSON.stringify(updateOrderDto)}`);
      const order = await this.ordersService.update(orderId, updateOrderDto, user);
      this.logger.log(`PATCH /api/orders/${id} - Successfully updated order: ${order.orderNumber}`);
      return {
        statusCode: HttpStatus.OK,
        message: 'Order updated successfully',
        data: order,
      };
    } catch (error) {
      const errorType = error.constructor.name;
      const errorMessage = error.message || 'Unknown error';
      this.logger.error(`PATCH /api/orders/${id} - Error updating order. Error Type: ${errorType}, Message: ${errorMessage}`, error.stack);
      throw error;
    }
  }

  @Delete(':id')
  async remove(@Param('id') id: string, @User() user: any) {
    try {
      const orderId = +id;
      this.logger.debug(`DELETE /api/orders/${id} - Deleting order with ID: ${orderId}, user ID: ${user.id}`);
      await this.ordersService.remove(orderId, user);
      this.logger.log(`DELETE /api/orders/${id} - Successfully deleted order`);
      return {
        statusCode: HttpStatus.OK,
        message: 'Order deleted successfully',
      };
    } catch (error) {
      const errorType = error.constructor.name;
      const errorMessage = error.message || 'Unknown error';
      this.logger.error(`DELETE /api/orders/${id} - Error deleting order. Error Type: ${errorType}, Message: ${errorMessage}`, error.stack);
      throw error;
    }
  }
}




