import { Injectable, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, FindOptionsOrder } from 'typeorm';
import { CustomerNote } from './entities/customer-note.entity';
import { Customer } from './entities/customer.entity';
import { User } from '../users/entities/user.entity';
import { CreateCustomerNoteDto } from './dto/create-customer-note.dto';

@Injectable()
export class CustomerNotesService {
  constructor(
    @InjectRepository(CustomerNote)
    private customerNotesRepository: Repository<CustomerNote>,
    @InjectRepository(Customer)
    private customersRepository: Repository<Customer>,
    @InjectRepository(User)
    private usersRepository: Repository<User>,
  ) {}

async create(createCustomerNoteDto: CreateCustomerNoteDto, user: any): Promise<CustomerNote> {
  const { note, customer: customerId, parent_note: parentNoteId, child_note_id, isActive } = createCustomerNoteDto;
  try {
    const customer = await this.customersRepository.findOne({ where: { id: customerId } });
    if (!customer) throw new NotFoundException(`Customer with ID ${customerId} not found`);

    let parentNote: CustomerNote | null = null;
    if (parentNoteId) {
      parentNote = await this.customerNotesRepository.findOne({ where: { id: parentNoteId } });
      if (!parentNote) throw new NotFoundException(`Parent note with ID ${parentNoteId} not found`);
    }

    let createdBy: User | null = null;
    if (user && user.id) {
      createdBy = await this.usersRepository.findOne({ where: { id: user.id } });
      if (!createdBy) throw new NotFoundException(`Created by user with ID ${user.id} not found`);
    }

    let childNote: CustomerNote | null = null;
    if (child_note_id) {
      childNote = await this.customerNotesRepository.findOne({ where: { id: child_note_id } });
      if (!childNote) throw new NotFoundException(`Child note with ID ${child_note_id} not found`);
    }

    const newNote = this.customerNotesRepository.create({
      note,
      customer,
      parentNote,
      childNote,
      createdBy,
      isActive: typeof isActive !== 'undefined' ? isActive : true,
    });
    const savedNote = await this.customerNotesRepository.save(newNote);
    return savedNote;
  } catch (error) {
    throw error;
  }
}


async findAll(
  customerId: number,
  ordering: string = '-created_at',
  limit: number = 10,
  offset: number = 0,
  user: any
): Promise<{
  data: any[]; count: number; next: string | null; previous: string | null;
}> {
  try {
    const userEntity = await this.usersRepository.findOne({ where: { id: user.id } });
    if (!userEntity) throw new NotFoundException(`User with ID ${user.id} not found`);

    const order: FindOptionsOrder<CustomerNote> = {
      createdAt: ordering === '-created_at' ? 'DESC' : 'ASC',
    };

    const [notes, total] = await this.customerNotesRepository.findAndCount({
      where: { customer: { id: customerId } },
      order,
      relations: ['parentNote', 'childNote', 'customer', 'createdBy'],
      skip: offset,
      take: limit,
    });

    const transformedNotes = notes.map(note => ({
      id: note.id,
      created_at: note.createdAt,
      updated_at: note.updatedAt,
      is_active: note.isActive,
      note: note.note,
      child_note_id: note.childNote ? note.childNote.id : null,
      created_by_id: note.createdBy ? note.createdBy.id : null,
      created_by_name: note.createdBy ? `${note.createdBy.firstName} ${note.createdBy.lastName}`.trim() : null,
      customer_id: note.customer.id,
      parent_note_id: note.parentNote ? note.parentNote.id : null,
    }));

    const baseUrl = '/api/customers/notes';
    const next =
      offset + limit < total
        ? `${baseUrl}?customer=${customerId}&offset=${offset + limit}&limit=${limit}&ordering=${ordering}`
        : null;
    const previous =
      offset > 0
        ? `${baseUrl}?customer=${customerId}&offset=${Math.max(0, offset - limit)}&limit=${limit}&ordering=${ordering}`
        : null;

    return { data: transformedNotes, count: total, next, previous };
  } catch (error) {
    throw error;
  }
}

}
