import { Test, TestingModule } from '@nestjs/testing';
import { UsersService } from './users.service';
import { getRepositoryToken } from '@nestjs/typeorm';
import { User } from './entities/user.entity';
import { Repository, DataSource } from 'typeorm';
import { CreateUserDto } from './dto/create-user.dto';
import { UpdateUserDto } from './dto/update-user.dto';
import { NotFoundException } from '@nestjs/common';
import { EmailService } from '../common/services/email.service';
import { UserActivity } from '../user-activity/entities/user-activity.entity';
import { Login } from '../login/entities/login.entity';
import { PermissionsService } from '../permissions/permissions.service';

describe('UsersService', () => {
  let service: UsersService;
  let repository: Repository<User>;

  const mockRepository = {
    create: jest.fn(),
    save: jest.fn(),
    find: jest.fn(),
    findOne: jest.fn(),
    update: jest.fn(),
    delete: jest.fn(),
    createQueryBuilder: jest.fn(),
  };

  const mockActivityRepository = {
    create: jest.fn(),
    save: jest.fn(),
  };

  const mockLoginRepository = {
    create: jest.fn(),
    save: jest.fn(),
    findOne: jest.fn(),
  };

  const mockEmailService = {
    sendWelcomeEmail: jest.fn().mockResolvedValue(undefined),
    sendReportsToNotification: jest.fn().mockResolvedValue(undefined),
  };

  const mockPermissionsService = {
    findAll: jest.fn().mockResolvedValue({
      data: [
        { id: 89, name: 'Permission 1', codename: 'perm_1' },
        { id: 90, name: 'Permission 2', codename: 'perm_2' },
      ],
    }),
  };

  const mockDataSource = {
    transaction: jest.fn((callback) => callback({
      findOne: jest.fn(),
      create: jest.fn(),
      save: jest.fn(),
    })),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        UsersService,
        {
          provide: getRepositoryToken(User),
          useValue: mockRepository,
        },
        {
          provide: getRepositoryToken(UserActivity),
          useValue: mockActivityRepository,
        },
        {
          provide: getRepositoryToken(Login),
          useValue: mockLoginRepository,
        },
        {
          provide: EmailService,
          useValue: mockEmailService,
        },
        {
          provide: PermissionsService,
          useValue: mockPermissionsService,
        },
        {
          provide: DataSource,
          useValue: mockDataSource,
        },
      ],
    }).compile();

    service = module.get<UsersService>(UsersService);
    repository = module.get<Repository<User>>(getRepositoryToken(User));
  });

  it('should be defined', () => {
    expect(service).toBeDefined();
  });

  describe('create', () => {
    it('should create a new user', async () => {
      const createUserDto: CreateUserDto = {
        user: { email: 'test@example.com', name: 'John Doe' },
        role_id: '6',
        permissions: ['89', '90'],
        reports_to_id: '2',
      };

      const user = {
        id: 1,
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        role: 'Agent',
        isActive: true,
        status: 1,
        createdAt: new Date(),
        updatedAt: new Date(),
      };

      const reportsToUser = {
        id: 2,
        email: 'manager@example.com',
        firstName: 'Manager',
        lastName: 'User',
        role: 'Team Lead',
        isActive: true,
        status: 1,
        createdAt: new Date(),
        updatedAt: new Date(),
      };

      const expectedResponse = {
        data: {
          id: 1,
          user: {
            email: 'test@example.com',
            name: 'John Doe',
            last_login: null,
          },
          granted_permissions: [
            { id: 89, name: 'Permission 1', codename: 'perm_1' },
            { id: 90, name: 'Permission 2', codename: 'perm_2' },
          ],
          role: {
            id: 6,
            name: 'Agent',
            permissions: [
              { id: 89, name: 'Permission 1', codename: 'perm_1' },
              { id: 90, name: 'Permission 2', codename: 'perm_2' },
            ],
          },
          reports_to: {
            id: 2,
            user: {
              email: 'manager@example.com',
              name: 'Manager User',
              last_login: null,
            },
            granted_permissions: [],
          },
          created_at: user.createdAt,
          updated_at: user.updatedAt,
          is_active: true,
        },
        success: true,
        message: 'ok',
        status: 201,
      };

      // Mock the transaction callback
      const mockTransactionManager = {
        findOne: jest.fn()
          .mockResolvedValueOnce(null) // No existing user
          .mockResolvedValueOnce(reportsToUser) // Reports to user
          .mockResolvedValueOnce({ ...user, reportsTo: reportsToUser }), // User with reportsTo
        create: jest.fn().mockReturnValue(user),
        save: jest.fn().mockResolvedValue(user),
      };

      mockDataSource.transaction.mockImplementation((callback) => 
        callback(mockTransactionManager)
      );

      // Mock permissionsService.findAll to return all arrays
      jest.spyOn(mockPermissionsService, 'findAll').mockResolvedValue({
        data: [
          { id: 89, name: 'Permission 1', codename: 'perm_1' },
          { id: 90, name: 'Permission 2', codename: 'perm_2' },
        ],
        customer_permissions: [],
        user_permissions: [],
      });

      const result = await service.create(createUserDto, 1);
      expect(result).toEqual(expectedResponse);
      expect(mockDataSource.transaction).toHaveBeenCalled();
      expect(mockEmailService.sendWelcomeEmail).toHaveBeenCalledWith('test@example.com');
    });
  });

  describe('findAll', () => {
    it('should return an array of users', async () => {
      const expectedUsers = [
        {
          id: 1,
          email: 'test@example.com',
          firstName: 'John',
          lastName: 'Doe',
          role: 'Agent',
          isActive: true,
          status: 1,
          createdAt: new Date(),
          updatedAt: new Date(),
          reportsTo: null,
        },
      ];

      const expectedResponse = {
        next: null,
        previous: null,
        count: 1,
        limit: 10,
        offset: 0,
        status: 200,
        success: true,
        message: 'List Successfully Retrieved!',
        data: [
          {
            id: 1,
            user: {
              email: 'test@example.com',
              name: 'John Doe',
              last_login: null,
            },
            role: {
              id: 6,
              name: 'Agent',
              permissions: [],
            },
            reports_to: null,
            granted_permissions: [],
            is_active: true,
          },
        ],
      };

      const mockQueryBuilder = {
        leftJoinAndSelect: jest.fn().mockReturnThis(),
        where: jest.fn().mockReturnThis(),
        andWhere: jest.fn().mockReturnThis(),
        getCount: jest.fn().mockResolvedValue(1),
        skip: jest.fn().mockReturnThis(),
        take: jest.fn().mockReturnThis(),
        getMany: jest.fn().mockResolvedValue(expectedUsers),
      };

      mockRepository.createQueryBuilder.mockReturnValue(mockQueryBuilder);
      mockLoginRepository.findOne.mockResolvedValue(null);

      const result = await service.findAll();
      expect(result).toEqual(expectedResponse);
      expect(mockRepository.createQueryBuilder).toHaveBeenCalledWith('user');
    });
  });

  describe('findOne', () => {
    it('should return a single user', async () => {
      const expectedUser = {
        id: 1,
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        role: 'Agent',
        isActive: true,
        status: 1,
        createdAt: new Date(),
        updatedAt: new Date(),
        reportsTo: null,
      };

      const expectedResponse = {
        id: 1,
        user: {
          email: 'test@example.com',
          name: 'John Doe',
          last_login: null,
        },
        granted_permissions: [],
        role: {
          id: 6,
          name: 'Agent',
          permissions: [
            { id: 89, name: 'Permission 1', codename: 'perm_1' },
            { id: 90, name: 'Permission 2', codename: 'perm_2' },
          ],
        },
        reports_to: null,
        created_at: expectedUser.createdAt,
        updated_at: expectedUser.updatedAt,
        is_active: true,
      };

      mockRepository.findOne.mockResolvedValue(expectedUser);

      const result = await service.findOne(1);
      expect(result).toEqual(expectedResponse);
      expect(repository.findOne).toHaveBeenCalledWith({ 
        where: { id: 1 },
        relations: ['reportsTo']
      });
    });

    it('should throw NotFoundException if user not found', async () => {
      mockRepository.findOne.mockResolvedValue(null);
      await expect(service.findOne(5)).rejects.toThrow(new NotFoundException('User with ID 5 not found'));
    });
  });

  describe('update', () => {
    it('should update a user', async () => {
      const updateUserDto: UpdateUserDto = {
        user: { email: 'test@example.com', name: 'Updated Name' },
      };

      const existingUser = {
        id: 1,
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        role: 'Agent',
        isActive: true,
        status: 1,
        createdAt: new Date(),
        updatedAt: new Date(),
        reportsTo: null,
      };

      const updatedUser = {
        ...existingUser,
        firstName: 'Updated',
        lastName: 'Name',
      };

      const expectedResponse = {
        id: 1,
        user: {
          email: 'test@example.com',
          name: 'Updated Name',
          last_login: null,
        },
        granted_permissions: [],
        role: {
          id: 6,
          name: 'Agent',
          permissions: [
            { id: 89, name: 'Permission 1', codename: 'perm_1' },
            { id: 90, name: 'Permission 2', codename: 'perm_2' },
          ],
        },
        reports_to: null,
        created_at: updatedUser.createdAt,
        updated_at: updatedUser.updatedAt,
        is_active: true,
      };

      // Mock the transaction callback
      const mockTransactionManager = {
        findOne: jest.fn().mockResolvedValue(existingUser),
        save: jest.fn().mockResolvedValue(updatedUser),
      };

      mockDataSource.transaction.mockImplementation((callback) => 
        callback(mockTransactionManager)
      );

      const result = await service.update(1, updateUserDto);
      expect(result).toEqual(expectedResponse);
      expect(mockDataSource.transaction).toHaveBeenCalled();
    });

    it('should throw NotFoundException if user not found', async () => {
      // Mock the transaction callback
      const mockTransactionManager = {
        findOne: jest.fn().mockResolvedValue(null),
      };

      mockDataSource.transaction.mockImplementation((callback) => 
        callback(mockTransactionManager)
      );

      await expect(service.update(5, { user: { email: 'test@example.com', name: 'Updated Name' } })).rejects.toThrow(new NotFoundException('User with ID 5 not found'));
    });
  });

  describe('remove', () => {
    it('should remove a user', async () => {
      const existingUser = {
        id: 1,
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        role: 'Agent',
        isActive: true,
        status: 1,
        createdAt: new Date(),
        updatedAt: new Date(),
        reportsTo: null,
      };

      mockRepository.findOne.mockResolvedValue(existingUser);
      mockRepository.save.mockResolvedValue({ ...existingUser, isActive: false });

      const result = await service.remove(1);
      expect(result).toEqual({ status: 'success', message: 'User has been successfully deactivated' });
      expect(repository.findOne).toHaveBeenCalledWith({ where: { id: 1 } });
      expect(repository.save).toHaveBeenCalled();
    });

    it('should throw NotFoundException if user not found', async () => {
      mockRepository.findOne.mockResolvedValue(null);
      await expect(service.remove(5)).rejects.toThrow(new NotFoundException('User with ID 5 not found'));
    });
  });

  describe('deactivate', () => {
    it('should deactivate a user', async () => {
      const user = { 
        id: 1, 
        isActive: true,
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        role: 'Agent',
        status: 1,
        createdAt: new Date(),
        updatedAt: new Date(),
      };
      
      // Mock findOne to return the user object that the service expects
      mockRepository.findOne.mockResolvedValue(user);
      mockRepository.save.mockResolvedValue({ ...user, isActive: false });

      const result = await service.deactivate(1);
      expect(result.user.isActive).toBe(false);
      expect(result.message).toBeDefined();
      expect(repository.findOne).toHaveBeenCalledWith({ 
        where: { id: 1 },
        relations: ['reportsTo']
      });
      expect(repository.save).toHaveBeenCalled();
    });

    it('should throw InternalServerErrorException if user not found', async () => {
      mockRepository.findOne.mockResolvedValue(null);
      await expect(service.deactivate(5)).rejects.toThrow('Failed to deactivate user');
    });
  });

  describe('reactivate', () => {
    it('should reactivate a user', async () => {
      const user = { 
        id: 1, 
        isActive: false,
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        role: 'Agent',
        status: 1,
        createdAt: new Date(),
        updatedAt: new Date(),
      };
      
      mockRepository.findOne.mockResolvedValue(user);
      mockRepository.save.mockResolvedValue({ ...user, isActive: true });

      const result = await service.reactivate(1);
      expect(result.user.isActive).toBe(true);
      expect(result.message).toBeDefined();
      expect(repository.findOne).toHaveBeenCalledWith({ where: { id: 1 } });
      expect(repository.save).toHaveBeenCalled();
    });

    it('should throw InternalServerErrorException if user not found', async () => {
      mockRepository.findOne.mockResolvedValue(null);
      await expect(service.reactivate(5)).rejects.toThrow('Failed to reactivate user');
    });
  });

  describe('findDeactivatedUsers', () => {
    it('should return deactivated users', async () => {
      const deactivatedUsers = [
        { id: 1, isActive: false, email: 'user1@example.com' },
        { id: 2, isActive: false, email: 'user2@example.com' },
      ];

      mockRepository.find.mockResolvedValue(deactivatedUsers);

      const result = await service.findDeactivatedUsers();
      expect(result).toEqual(deactivatedUsers);
      expect(repository.find).toHaveBeenCalledWith({ where: { isActive: false } });
    });
  });

  describe('bulkCreate', () => {
    it('should create users from a valid Excel file', async () => {
      const mockFile = {
        buffer: Buffer.from('test'),
        originalname: 'users.xlsx',
      } as Express.Multer.File;

      const expectedResponse = {
        message: 'Successfully created 2 users.',
        details: {
          success: 2,
          failed: 0,
          errors: [],
        },
      };

      // Mock XLSX.read to return mock data
      const mockWorkbook = {
        SheetNames: ['Sheet1'],
        Sheets: {
          Sheet1: {
            'A1': { v: 'email' },
            'B1': { v: 'firstName' },
            'C1': { v: 'lastName' },
            'D1': { v: 'role' },
            'A2': { v: 'user1@example.com' },
            'B2': { v: 'John' },
            'C2': { v: 'Doe' },
            'D2': { v: 'Agent' },
            'A3': { v: 'user2@example.com' },
            'B3': { v: 'Jane' },
            'C3': { v: 'Smith' },
            'D3': { v: 'Agent' },
          },
        },
      };

      // Mock XLSX.read
      const XLSX = require('xlsx');
      jest.spyOn(XLSX, 'read').mockReturnValue(mockWorkbook);
      jest.spyOn(XLSX.utils, 'sheet_to_json').mockReturnValue([
        { email: 'user1@example.com', firstName: 'John', lastName: 'Doe', role: 'Agent' },
        { email: 'user2@example.com', firstName: 'Jane', lastName: 'Smith', role: 'Agent' },
      ]);

      // Mock the create method to return success
      jest.spyOn(service, 'create').mockResolvedValue({
        data: { id: 1, user: { email: 'user1@example.com', name: 'John Doe' } },
        success: true,
        message: 'ok',
        status: 201,
      });

      const result = await service.bulkCreate(mockFile);
      expect(result).toEqual(expectedResponse);
    });

    it('should throw BadRequestException for invalid file', async () => {
      const mockFile = {
        buffer: Buffer.from('invalid'),
        originalname: 'users.txt',
      } as Express.Multer.File;

      // Mock XLSX.read to throw an error
      const XLSX = require('xlsx');
      jest.spyOn(XLSX, 'read').mockImplementation(() => {
        throw new Error('Invalid Excel file');
      });

      await expect(service.bulkCreate(mockFile)).rejects.toThrow('Invalid Excel file');
    });
  });
}); 