import { Test, TestingModule } from '@nestjs/testing';
import { RolesService } from './roles.service';
import { getRepositoryToken } from '@nestjs/typeorm';
import { Role } from './entities/role.entity';
import { Repository } from 'typeorm';
import { CreateRoleDto } from './dto/create-role.dto';
import { UpdateRoleDto } from './dto/update-role.dto';
import { NotFoundException } from '@nestjs/common';
import { PermissionsService } from '../permissions/permissions.service';

describe('RolesService', () => {
  let service: RolesService;
  let repository: Repository<Role>;
  let permissionsService: PermissionsService;

  const mockRepository = {
    create: jest.fn(),
    save: jest.fn(),
    find: jest.fn(),
    findOne: jest.fn(),
    update: jest.fn(),
    delete: jest.fn(),
  };

  const mockPermissionsService = {
    findAll: jest.fn(),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        RolesService,
        {
          provide: getRepositoryToken(Role),
          useValue: mockRepository,
        },
        {
          provide: PermissionsService,
          useValue: mockPermissionsService,
        },
      ],
    }).compile();

    service = module.get<RolesService>(RolesService);
    repository = module.get<Repository<Role>>(getRepositoryToken(Role));
    permissionsService = module.get<PermissionsService>(PermissionsService);
  });

  it('should be defined', () => {
    expect(service).toBeDefined();
  });

  // describe('create', () => {
  //   it('should create a new role', async () => {
  //     const roleData: CreateRoleDto = {
  //       name: 'Test Role',
  //       description: 'Test Description',
  //     };
  //     const expectedRole = {
  //       id: 1,
  //       ...roleData,
  //       createdAt: new Date(),
  //       updatedAt: new Date(),
  //     };
  //     mockRepository.create.mockReturnValue(expectedRole);
  //     mockRepository.save.mockResolvedValue(expectedRole);
  //     const result = await service.create(roleData);
  //     expect(result).toEqual(expectedRole);
  //     expect(repository.create).toHaveBeenCalledWith(roleData);
  //     expect(repository.save).toHaveBeenCalled();
  //   });
  // });

  describe('findAll', () => {
    it('should return an array of roles', async () => {
      const mockRoles = [
        {
          id: 1,
          name: 'Test Role',
          description: 'Test Description',
          createdAt: new Date(),
          updatedAt: new Date(),
        },
      ];

      const mockPermissionsResponse = {
        data: [
          {
            id: 1,
            name: 'Can View Test',
            codename: 'view_test',
            content_type: 0,
          },
        ],
        success: true,
        message: 'List Successfully Retrieved!',
        status: 200,
      };

      const expectedResponse = {
        data: [
          {
            id: 1,
            name: 'Test Role',
            permissions: mockPermissionsResponse.data,
          },
        ],
        success: true,
        message: 'List Successfully Retrieved!',
        status: 200,
      };

      mockRepository.find.mockResolvedValue(mockRoles);
      mockPermissionsService.findAll.mockResolvedValue(mockPermissionsResponse);

      const result = await service.findAll();
      expect(result).toEqual(expectedResponse);
      expect(repository.find).toHaveBeenCalled();
      expect(permissionsService.findAll).toHaveBeenCalledWith('1');
    });
  });

  describe('findOne', () => {
    it('should return a single role', async () => {
      const expectedRole = {
        id: 1,
        name: 'Test Role',
        description: 'Test Description',
        createdAt: new Date(),
        updatedAt: new Date(),
      };
      mockRepository.findOne.mockResolvedValue(expectedRole);
      const result = await service.findOne(1);
      expect(result).toEqual(expectedRole);
      expect(repository.findOne).toHaveBeenCalledWith({ where: { id: 1 } });
    });

    it('should throw NotFoundException if role not found', async () => {
      mockRepository.findOne.mockResolvedValue(null);
      await expect(service.findOne(999)).rejects.toThrow(NotFoundException);
    });
  });

  // describe('update', () => {
  //   it('should update a role', async () => {
  //     const updateData: UpdateRoleDto = {
  //       name: 'Updated Role',
  //     };
  //     const existingRole = {
  //       id: 1,
  //       name: 'Test Role',
  //       description: 'Test Description',
  //       createdAt: new Date(),
  //       updatedAt: new Date(),
  //     };
  //     const updatedRole = {
  //       ...existingRole,
  //       ...updateData,
  //     };
  //     mockRepository.findOne.mockResolvedValue(existingRole);
  //     mockRepository.save.mockResolvedValue(updatedRole);
  //     const result = await service.update(1, updateData);
  //     expect(result).toEqual(updatedRole);
  //     expect(repository.findOne).toHaveBeenCalledWith({ where: { id: 1 } });
  //     expect(repository.save).toHaveBeenCalled();
  //   });
  //   it('should throw NotFoundException if role not found', async () => {
  //     mockRepository.findOne.mockResolvedValue(null);
  //     await expect(service.update(999, { name: 'Updated Role' })).rejects.toThrow(NotFoundException);
  //   });
  // });

  // describe('remove', () => {
  //   it('should remove a role', async () => {
  //     const existingRole = {
  //       id: 1,
  //       name: 'Test Role',
  //       description: 'Test Description',
  //       createdAt: new Date(),
  //       updatedAt: new Date(),
  //     };
  //     mockRepository.findOne.mockResolvedValue(existingRole);
  //     mockRepository.delete.mockResolvedValue({ affected: 1 });
  //     const result = await service.remove(1);
  //     expect(result).toEqual({ status: 'success', message: 'Role has been successfully deleted' });
  //     expect(repository.findOne).toHaveBeenCalledWith({ where: { id: 1 } });
  //     expect(repository.delete).toHaveBeenCalledWith(1);
  //   });
  //   it('should throw NotFoundException if role not found', async () => {
  //     mockRepository.findOne.mockResolvedValue(null);
  //     await expect(service.remove(999)).rejects.toThrow(NotFoundException);
  //   });
  // });
}); 