import { Injectable, NotFoundException, ConflictException, BadRequestException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, Like } from 'typeorm';
import { Product } from './entities/product.entity';
import { CreateProductDto } from './dto/create-product.dto';
import { UpdateProductDto } from './dto/update-product.dto';
import { ListProductsDto } from './dto/list-products.dto';

@Injectable()
export class ProductsService {
  constructor(
    @InjectRepository(Product)
    private readonly productsRepository: Repository<Product>,
  ) {}

  async create(createProductDto: CreateProductDto): Promise<Product> {
    try {
      // Check if product with same name already exists
      const existingProduct = await this.productsRepository.findOne({
        where: { name: createProductDto.name },
      });

      if (existingProduct) {
        throw new ConflictException(`Product with name "${createProductDto.name}" already exists`);
      }

      const product = this.productsRepository.create(createProductDto);
      return await this.productsRepository.save(product);
    } catch (error) {
      if (error instanceof ConflictException) {
        throw error;
      }
      throw new BadRequestException(`Failed to create product: ${error.message}`);
    }
  }

  async findAll(listProductsDto: ListProductsDto): Promise<{ data: Product[]; total: number }> {
    const { page = 1, limit = 10, search, isActive } = listProductsDto;
    const skip = (page - 1) * limit;

    const queryBuilder = this.productsRepository.createQueryBuilder('product');

    if (search) {
      queryBuilder.where(
        '(product.name LIKE :search OR product.description LIKE :search)',
        { search: `%${search}%` },
      );
    }

    if (isActive !== undefined) {
      if (search) {
        queryBuilder.andWhere('product.isActive = :isActive', { isActive });
      } else {
        queryBuilder.where('product.isActive = :isActive', { isActive });
      }
    }

    queryBuilder.skip(skip).take(limit).orderBy('product.createdAt', 'DESC');

    const [data, total] = await queryBuilder.getManyAndCount();

    return { data, total };
  }

  async findOne(id: number): Promise<Product> {
    const product = await this.productsRepository.findOne({
      where: { id },
    });

    if (!product) {
      throw new NotFoundException(`Product with ID ${id} not found`);
    }

    return product;
  }

  async update(id: number, updateProductDto: UpdateProductDto): Promise<Product> {
    const product = await this.findOne(id);

    // Check if name is being updated and if it conflicts with existing product
    if (updateProductDto.name && updateProductDto.name !== product.name) {
      const existingProduct = await this.productsRepository.findOne({
        where: { name: updateProductDto.name },
      });

      if (existingProduct) {
        throw new ConflictException(`Product with name "${updateProductDto.name}" already exists`);
      }
    }

    Object.assign(product, updateProductDto);
    return await this.productsRepository.save(product);
  }

  async remove(id: number): Promise<void> {
    const product = await this.findOne(id);
    await this.productsRepository.remove(product);
  }

  async seedProducts(): Promise<void> {
    try {
      const existingProducts = await this.productsRepository.count();
      if (existingProducts === 0) {
        const defaultProducts = [
          {
            name: 'Premium CRM License',
            description: 'Full-featured CRM license with unlimited users and advanced analytics',
          },
          {
            name: 'Basic CRM Package',
            description: 'Entry-level CRM package with essential features for small teams',
          },
          {
            name: 'Enterprise Support Plan',
            description: '24/7 priority support with dedicated account manager',
          },
          {
            name: 'Data Migration Service',
            description: 'Professional data migration service from legacy systems',
          },
          {
            name: 'Custom Integration Package',
            description: 'Tailored integration solutions for third-party applications',
          },
        ];

        for (const productData of defaultProducts) {
          const existing = await this.productsRepository.findOne({
            where: { name: productData.name },
          });
          if (!existing) {
            await this.productsRepository.save(productData);
          }
        }
      }
    } catch (error) {
      // Silently fail if table doesn't exist yet - it will be created by TypeORM synchronize
      // or migrations, and seeding can be done manually later
      console.warn('Products seeding skipped - table may not exist yet:', error.message);
    }
  }
}

