import { Test, TestingModule } from '@nestjs/testing';
import { PermissionsService } from './permissions.service';
import { getRepositoryToken } from '@nestjs/typeorm';
import { Permission } from './entities/permission.entity';
import { Repository } from 'typeorm';
import { NotFoundException } from '@nestjs/common';

describe('PermissionsService', () => {
  let service: PermissionsService;
  let repository: Repository<Permission>;

  const mockRepository = {
    find: jest.fn(),
    findOne: jest.fn(),
    createQueryBuilder: jest.fn(() => ({
      where: jest.fn().mockReturnThis(),
      orderBy: jest.fn().mockReturnThis(),
      addOrderBy: jest.fn().mockReturnThis(),
      getMany: jest.fn(),
    })),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        PermissionsService,
        {
          provide: getRepositoryToken(Permission),
          useValue: mockRepository,
        },
      ],
    }).compile();

    service = module.get<PermissionsService>(PermissionsService);
    repository = module.get<Repository<Permission>>(getRepositoryToken(Permission));
  });

  it('should be defined', () => {
    expect(service).toBeDefined();
  });

  describe('findAll', () => {
    it('should return an array of permissions', async () => {
      const mockPermissions = [
        {
          id: 1,
          name: 'Test Permission',
          description: 'Test Description',
          category: 'Test Category',
          entity: 'Test Entity',
          action: 'Can View Test Entity',
          createdAt: new Date(),
          updatedAt: new Date(),
        },
      ];

      const expectedResponse = {
        data: [],
        customer_permissions: [],
        user_permissions: [],
        success: true,
        message: '',
        status: 200,
      };

      mockRepository.find.mockResolvedValue(mockPermissions);

      const result = await service.findAll();
      expect(result).toEqual(expectedResponse);
      expect(repository.find).toHaveBeenCalled();
    });
  });

  describe('findByCategory', () => {
    it('should return permissions by category', async () => {
      const category = 'Test Category';
      const mockPermissions = [
        {
          id: 1,
          name: 'Test Permission',
          description: 'Test Description',
          category: category,
          entity: 'Test Entity',
          action: 'Can View Test Entity',
          createdAt: new Date(),
          updatedAt: new Date(),
        },
      ];

      const expectedResponse = {
        data: [
          {
            id: 1,
            name: 'Can View Test Entity',
            codename: 'view_testentity',
            content_type: 0,
          },
        ],
        success: true,
        message: 'List Successfully Retrieved!',
        status: 200,
      };

      mockRepository.find.mockResolvedValue(mockPermissions);

      const result = await service.findByCategory(category);
      expect(result).toEqual(expectedResponse);
      expect(repository.find).toHaveBeenCalledWith({
        where: { category },
        order: {
          entity: 'ASC',
          action: 'ASC',
        },
      });
    });
  });

  describe('findOne', () => {
    it('should return a single permission', async () => {
      const expectedPermission = {
        id: 1,
        name: 'Test Permission',
        description: 'Test Description',
        category: 'Test Category',
        entity: 'Test Entity',
        action: 'Can View Test Entity',
        createdAt: new Date(),
        updatedAt: new Date(),
      };

      mockRepository.findOne.mockResolvedValue(expectedPermission);

      const result = await service.findOne(1);
      expect(result).toEqual(expectedPermission);
      expect(repository.findOne).toHaveBeenCalledWith({ where: { id: 1 } });
    });

    it('should return null if permission not found', async () => {
      mockRepository.findOne.mockResolvedValue(null);
      const result = await service.findOne(999);
      expect(result).toBeNull();
    });
  });
}); 