import {
  Injectable,
  NotFoundException,
  BadRequestException,
  ConflictException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, Between, Like, DataSource } from 'typeorm';
import { Order } from './entities/order.entity';
import { OrderItem } from './entities/order-item.entity';
import { Customer } from '../customers/entities/customer.entity';
import { Product } from '../products/entities/product.entity';
import { User } from '../users/entities/user.entity';
import { CreateOrderDto } from './dto/create-order.dto';
import { UpdateOrderDto } from './dto/update-order.dto';
import { ListOrdersDto } from './dto/list-orders.dto';

@Injectable()
export class OrdersService {
  constructor(
    @InjectRepository(Order)
    private readonly ordersRepository: Repository<Order>,
    @InjectRepository(OrderItem)
    private readonly orderItemsRepository: Repository<OrderItem>,
    @InjectRepository(Customer)
    private readonly customersRepository: Repository<Customer>,
    @InjectRepository(Product)
    private readonly productsRepository: Repository<Product>,
    @InjectRepository(User)
    private readonly usersRepository: Repository<User>,
    private readonly dataSource: DataSource,
  ) {}

  private async generateOrderNumber(): Promise<string> {
    const today = new Date();
    const dateStr = today.toISOString().split('T')[0].replace(/-/g, '');
    
    // Find the last order number for today (including null/empty ones)
    const lastOrder = await this.ordersRepository
      .createQueryBuilder('order')
      .where('order.orderNumber LIKE :pattern', { pattern: `ORD-${dateStr}-%` })
      .orderBy('order.createdAt', 'DESC')
      .getOne();

    let sequence = 1;
    if (lastOrder && lastOrder.orderNumber) {
      const parts = lastOrder.orderNumber.split('-');
      if (parts.length >= 3) {
        const lastSequence = parseInt(parts[2] || '0', 10);
        sequence = lastSequence + 1;
      }
    }

    const orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
    
    // Double-check uniqueness
    const exists = await this.ordersRepository.findOne({
      where: { orderNumber },
    });
    
    if (exists) {
      // If somehow exists, increment and try again
      sequence++;
      return `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
    }

    return orderNumber;
  }

  async cleanupInvalidOrders(): Promise<void> {
    try {
      // First, fix empty order numbers
      await this.fixEmptyOrderNumbers();
      
      // Then, remove orders with invalid customerIds
      const allOrders = await this.ordersRepository.find();
      const allCustomerIds = (await this.customersRepository.find()).map(c => c.id);
      
      const invalidOrders = allOrders.filter(order => 
        order.customerId !== null && order.customerId !== undefined && !allCustomerIds.includes(order.customerId)
      );
      
      if (invalidOrders.length > 0) {
        console.log(`Found ${invalidOrders.length} orders with invalid customerIds. Removing...`);
        
        // Delete order items first (cascade should handle this, but being explicit)
        for (const order of invalidOrders) {
          await this.orderItemsRepository.delete({ orderId: order.id });
        }
        
        // Then delete the orders
        await this.ordersRepository.remove(invalidOrders);
        console.log(`Removed ${invalidOrders.length} orders with invalid customerIds.`);
      }
    } catch (error) {
      console.warn('Failed to cleanup invalid orders:', error.message);
    }
  }

  async fixEmptyOrderNumbers(): Promise<void> {
    try {
      // Find all orders with empty or null orderNumber
      const ordersWithoutNumber = await this.ordersRepository
        .createQueryBuilder('order')
        .where('order.orderNumber IS NULL OR order.orderNumber = :empty', { empty: '' })
        .getMany();

      if (ordersWithoutNumber.length > 0) {
        console.log(`Found ${ordersWithoutNumber.length} orders without order numbers. Fixing...`);
        
        for (const order of ordersWithoutNumber) {
          // Generate order number based on creation date
          const orderDate = order.createdAt || new Date();
          const dateStr = orderDate.toISOString().split('T')[0].replace(/-/g, '');
          
          // Find the highest sequence for that date (including all orders, not just existing ones)
          const allOrders = await this.ordersRepository
            .createQueryBuilder('o')
            .where('o.orderNumber IS NOT NULL AND o.orderNumber != :empty', { empty: '' })
            .andWhere('o.orderNumber LIKE :pattern', { pattern: `ORD-${dateStr}-%` })
            .getMany();
          
          let maxSequence = 0;
          allOrders.forEach(o => {
            if (o.orderNumber) {
              const parts = o.orderNumber.split('-');
              if (parts.length >= 3) {
                const seq = parseInt(parts[2] || '0', 10);
                if (seq > maxSequence) maxSequence = seq;
              }
            }
          });
          
          // Generate unique order number
          let sequence = maxSequence + 1;
          let orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
          
          // Ensure uniqueness by checking and incrementing if needed
          let exists = await this.ordersRepository.findOne({
            where: { orderNumber },
          });
          
          while (exists) {
            sequence++;
            orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
            exists = await this.ordersRepository.findOne({
              where: { orderNumber },
            });
          }
          
          order.orderNumber = orderNumber;
          await this.ordersRepository.save(order);
        }
        
        console.log(`Fixed ${ordersWithoutNumber.length} orders with missing order numbers.`);
      }
    } catch (error) {
      console.warn('Failed to fix empty order numbers:', error.message);
    }
  }

  async create(createOrderDto: CreateOrderDto, user: User): Promise<Order> {
    // Validate customer exists
    const customer = await this.customersRepository.findOne({
      where: { id: createOrderDto.customerId },
    });

    if (!customer) {
      throw new NotFoundException(`Customer with ID ${createOrderDto.customerId} not found`);
    }

    // Validate all products exist
    const productIds = createOrderDto.orderItems.map((item) => item.productId);
    const products = await this.productsRepository.find({
      where: productIds.map((id) => ({ id })),
    });

    if (products.length !== productIds.length) {
      const foundIds = products.map((p) => p.id);
      const missingIds = productIds.filter((id) => !foundIds.includes(id));
      throw new NotFoundException(
        `Products with IDs ${missingIds.join(', ')} not found`,
      );
    }

    // Generate unique order number (required, cannot be null)
    const orderNumber = await this.generateOrderNumber();
    if (!orderNumber) {
      throw new BadRequestException('Failed to generate order number');
    }

    // Validate customerId is provided
    if (!createOrderDto.customerId) {
      throw new BadRequestException('Customer ID is required');
    }

    // Create order
    const order = this.ordersRepository.create({
      orderNumber,
      customerId: createOrderDto.customerId,
      createdById: user.id,
    });

    const savedOrder = await this.ordersRepository.save(order);

    // Create order items
    const orderItems = createOrderDto.orderItems.map((item) =>
      this.orderItemsRepository.create({
        orderId: savedOrder.id,
        productId: item.productId,
        count: item.count,
      }),
    );

    await this.orderItemsRepository.save(orderItems);

    // Return order with relations
    return this.findOne(savedOrder.id, user);
  }

  async findAll(
    listOrdersDto: ListOrdersDto,
    user: User,
  ): Promise<{ data: Order[]; total: number }> {
    const {
      page = 1,
      limit = 10,
      customerId,
      orderNumber,
      createdById,
      startDate,
      endDate,
    } = listOrdersDto;
    const skip = (page - 1) * limit;

    const queryBuilder = this.ordersRepository
      .createQueryBuilder('order')
      .leftJoinAndSelect('order.customer', 'customer')
      .leftJoinAndSelect('order.createdBy', 'createdBy')
      .leftJoinAndSelect('order.orderItems', 'orderItems')
      .leftJoinAndSelect('orderItems.product', 'product');

    if (customerId) {
      queryBuilder.where('order.customerId = :customerId', { customerId });
    }

    if (orderNumber) {
      if (customerId) {
        queryBuilder.andWhere('order.orderNumber LIKE :orderNumber', {
          orderNumber: `%${orderNumber}%`,
        });
      } else {
        queryBuilder.where('order.orderNumber LIKE :orderNumber', {
          orderNumber: `%${orderNumber}%`,
        });
      }
    }

    if (createdById) {
      const whereClause = customerId || orderNumber ? 'andWhere' : 'where';
      queryBuilder[whereClause]('order.createdById = :createdById', {
        createdById,
      });
    }

    if (startDate && endDate) {
      const whereClause =
        customerId || orderNumber || createdById ? 'andWhere' : 'where';
      queryBuilder[whereClause]('order.createdAt BETWEEN :startDate AND :endDate', {
        startDate,
        endDate,
      });
    } else if (startDate) {
      const whereClause =
        customerId || orderNumber || createdById ? 'andWhere' : 'where';
      queryBuilder[whereClause]('order.createdAt >= :startDate', { startDate });
    } else if (endDate) {
      const whereClause =
        customerId || orderNumber || createdById ? 'andWhere' : 'where';
      queryBuilder[whereClause]('order.createdAt <= :endDate', { endDate });
    }

    queryBuilder
      .andWhere('order.isActive = :isActive', { isActive: true })
      .skip(skip)
      .take(limit)
      .orderBy('order.createdAt', 'DESC');

    const [data, total] = await queryBuilder.getManyAndCount();

    return { data, total };
  }

  async findOne(id: number, user: User): Promise<Order> {
    const order = await this.ordersRepository.findOne({
      where: { id },
      relations: ['customer', 'createdBy', 'orderItems', 'orderItems.product'],
    });

    if (!order) {
      throw new NotFoundException(`Order with ID ${id} not found`);
    }

    return order;
  }

  async update(
    id: number,
    updateOrderDto: UpdateOrderDto,
    user: User,
  ): Promise<Order> {
    // Use transaction to ensure atomicity
    return await this.dataSource.transaction(async (manager) => {
      const orderRepository = manager.getRepository(Order);
      const orderItemsRepository = manager.getRepository(OrderItem);
      const customersRepository = manager.getRepository(Customer);
      const productsRepository = manager.getRepository(Product);

      // Fetch order without orderItems relation to avoid cascade issues
      const order = await orderRepository.findOne({
        where: { id },
        relations: ['customer', 'createdBy'],
      });

      if (!order) {
        throw new NotFoundException(`Order with ID ${id} not found`);
      }

      // Update customer if provided
      if (updateOrderDto.customerId !== undefined) {
        const customer = await customersRepository.findOne({
          where: { id: updateOrderDto.customerId },
        });

        if (!customer) {
          throw new NotFoundException(
            `Customer with ID ${updateOrderDto.customerId} not found`,
          );
        }

        order.customerId = updateOrderDto.customerId;
      }

      // Update order items if provided
      if (updateOrderDto.orderItems !== undefined) {
        // Validate all products exist
        const productIds = updateOrderDto.orderItems.map((item) => item.productId);
        const products = await productsRepository.find({
          where: productIds.map((id) => ({ id })),
        });

        if (products.length !== productIds.length) {
          const foundIds = products.map((p) => p.id);
          const missingIds = productIds.filter((id) => !foundIds.includes(id));
          throw new NotFoundException(
            `Products with IDs ${missingIds.join(', ')} not found`,
          );
        }

        // Remove existing order items
        await orderItemsRepository.delete({ orderId: id });

        // Create new order items
        const orderItems = updateOrderDto.orderItems.map((item) =>
          orderItemsRepository.create({
            orderId: id,
            productId: item.productId,
            count: item.count,
          }),
        );

        await orderItemsRepository.save(orderItems);
      }

      // Update isActive if provided
      if (updateOrderDto.isActive !== undefined) {
        order.isActive = updateOrderDto.isActive;
      }

      await orderRepository.save(order);

      // Return order with all relations
      return this.findOne(id, user);
    });
  }

  async remove(id: number, user: User): Promise<void> {
    const order = await this.findOne(id, user);
    // Soft delete
    order.isActive = false;
    await this.ordersRepository.save(order);
  }
}

