import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  Query,
  HttpStatus,
  UseGuards,
} from '@nestjs/common';
import { OrdersService } from './orders.service';
import { CreateOrderDto } from './dto/create-order.dto';
import { UpdateOrderDto } from './dto/update-order.dto';
import { ListOrdersDto } from './dto/list-orders.dto';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';
import { PermissionsGuard } from '../auth/guards/permissions.guard';
import { RolesGuard } from '../auth/guards/roles.guard';
import { User } from '../auth/decorators/user.decorator';

@Controller('api/orders')
@UseGuards(JwtAuthGuard, PermissionsGuard, RolesGuard)
export class OrdersController {
  constructor(private readonly ordersService: OrdersService) {}

  @Post()
  async create(@Body() createOrderDto: CreateOrderDto, @User() user: any) {
    const order = await this.ordersService.create(createOrderDto, user);
    return {
      statusCode: HttpStatus.CREATED,
      message: 'Order created successfully',
      data: order,
    };
  }

  @Get()
  async findAll(@Query() listOrdersDto: ListOrdersDto, @User() user: any) {
    const { data, total } = await this.ordersService.findAll(listOrdersDto, user);
    return {
      statusCode: HttpStatus.OK,
      message: 'Orders retrieved successfully',
      data,
      total,
    };
  }

  @Get(':id')
  async findOne(@Param('id') id: string, @User() user: any) {
    const order = await this.ordersService.findOne(+id, user);
    return {
      statusCode: HttpStatus.OK,
      message: 'Order retrieved successfully',
      data: order,
    };
  }

  @Patch(':id')
  async update(
    @Param('id') id: string,
    @Body() updateOrderDto: UpdateOrderDto,
    @User() user: any,
  ) {
    const order = await this.ordersService.update(+id, updateOrderDto, user);
    return {
      statusCode: HttpStatus.OK,
      message: 'Order updated successfully',
      data: order,
    };
  }

  @Delete(':id')
  async remove(@Param('id') id: string, @User() user: any) {
    await this.ordersService.remove(+id, user);
    return {
      statusCode: HttpStatus.OK,
      message: 'Order deleted successfully',
    };
  }
}

