import * as XLSX from 'xlsx';
import { BadRequestException } from '@nestjs/common';
import { Injectable, ConflictException, NotFoundException, UnauthorizedException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, In, Like } from 'typeorm';
import { Customer } from './entities/customer.entity';
import { Address } from './entities/address.entity';
import { CustomerNote } from './entities/customer-note.entity';
import { User } from '../users/entities/user.entity';
import { CreateCustomerDto } from './dto/create-customer.dto';
import { UpdateCustomerDto } from './dto/update-customer.dto';
import { ListCustomersDto } from './dto/list-customers.dto';
import { BulkAssignCustomerDto } from './dto/bulk-assign-customer.dto';

interface ExcelCustomerRecord {
  id?: string | number;           // Customer ID (genCustomerId) - must be unique
  first_name?: string;
  last_name?: string;
  email?: string;
  mobile_phone?: string;          // Mobile number - must be unique
  home_phone?: string;
  title?: string;
  iso?: string;
  assigned_to_id?: string | number;
  // Address fields
  address1?: string;
  address2?: string;
  town_city?: string;
  county?: string;
  country?: string;
  post_code?: string;
  primary?: boolean | string;
  formatted_address?: string;
  [key: string]: any;
}

@Injectable()
export class CustomersService {
  constructor(
    @InjectRepository(Customer)
    private readonly customersRepository: Repository<Customer>,
    @InjectRepository(Address)
    private readonly addressesRepository: Repository<Address>,
    @InjectRepository(CustomerNote)
    private readonly customerNotesRepository: Repository<CustomerNote>,
    @InjectRepository(User)
    private readonly usersRepository: Repository<User>,
  ) {}

  // Allocate a unique customerId to prevent collisions after data migration
  private async allocateUniqueCustomerId(): Promise<string> {
    const pad = (n: number) => n.toString().padStart(2, '0');
    // Try a few times in the unlikely event of collision
    for (let i = 0; i < 10; i++) {
      const now = new Date();
      const y = now.getFullYear();
      const m = pad(now.getMonth() + 1);
      const d = pad(now.getDate());
      const rand = Math.random().toString(36).slice(2, 8).toUpperCase();
      const candidate = `CUST-${y}${m}${d}-${rand}`;
      const exists = await this.customersRepository.findOne({ where: { customerId: candidate } });
      if (!exists) return candidate;
    }
    // Fallback: long random if many collisions
    return `CUST-${Date.now()}-${Math.random().toString(36).slice(2).toUpperCase()}`;
  }

  async create(createCustomerDto: CreateCustomerDto, user: any): Promise<Customer> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    const { email, first_name, last_name, home_phone, mobile_phone, address, title, iso, DNC, is_land_line_default, assigned_to_id } = createCustomerDto;

    if (email) {
      const existingCustomer = await this.customersRepository.findOne({ where: { email } });
      if (existingCustomer) {
        throw new ConflictException('Email already exists');
      }
    }

    if (assigned_to_id) {
      const assignee = await this.usersRepository.findOne({ where: { id: parseInt(assigned_to_id.toString(), 10) } });
      if (!assignee) {
        throw new NotFoundException(`User with ID ${assigned_to_id} not found`);
      }
    }

    // Ensure AUTO_INCREMENT is ahead of current MAX(id) to avoid duplicate numeric IDs after migration
    try {
      const [maxRow] = await this.customersRepository.query(`SELECT COALESCE(MAX(id), 0) AS maxId FROM customers`);
      const [aiRow] = await this.customersRepository.query(`
        SELECT AUTO_INCREMENT AS nextId
        FROM INFORMATION_SCHEMA.TABLES
        WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'customers'
      `);
      const maxId = Number(maxRow?.maxId ?? 0);
      const nextId = Number(aiRow?.nextId ?? 1);
      if (nextId <= maxId) {
        await this.customersRepository.query(`ALTER TABLE customers AUTO_INCREMENT = ${maxId + 1}`);
      }
    } catch (_) {
      // ignore if no permission to read/alter; DB will still enforce PK uniqueness
    }

    const customer = this.customersRepository.create({
      firstName: first_name,
      lastName: last_name,
      email,
      mobileNumber: mobile_phone,
      landline: home_phone,
      title,
      iso,
      dnc: DNC || false,
      isLandLineDefault: is_land_line_default !== undefined ? is_land_line_default : true,
      assigned_to_id: assigned_to_id ? parseInt(assigned_to_id.toString(), 10) : user.id,
      createdById: user.id,
      isActive: true, // Explicitly set to ensure customers are visible
    });

    // Ensure a unique customerId on creation (prevents clashes with migrated data)
    customer.customerId = await this.allocateUniqueCustomerId();

    const savedCustomer = await this.customersRepository.save(customer);

    if (address && address.length > 0) {
      const addresses = address.map((addr) =>
        this.addressesRepository.create({
          ...addr,
          customer: savedCustomer,
          isActive: true,
        }),
      );
      await this.addressesRepository.save(addresses);
      savedCustomer.addresses = addresses;
    }

    return savedCustomer;
  }

  async update(id: number, updateCustomerDto: UpdateCustomerDto, user: any): Promise<Customer> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    console.log(`Updating customer with ID: ${id}`);

    const customer = await this.customersRepository.findOne({ where: { id }, relations: ['addresses'] });
    if (!customer) {
      throw new NotFoundException(`Customer with ID ${id} not found`);
    }

    const roleName = typeof user.role === 'string' ? user.role : user.role?.name || '';
    if (customer.assigned_to_id !== user.id && customer.createdById !== user.id && roleName.toLowerCase() !== 'admin' && roleName.toLowerCase() !== 'owner') {
      throw new UnauthorizedException('Not authorized to update this customer');
    }

    if (updateCustomerDto.email && updateCustomerDto.email !== customer.email) {
      const existingCustomer = await this.customersRepository.findOne({ where: { email: updateCustomerDto.email } });
      if (existingCustomer) {
        throw new ConflictException('Email already exists');
      }
    }

    if (updateCustomerDto.gen_customer_id && updateCustomerDto.gen_customer_id !== customer.genCustomerId) {
      const existingCustomer = await this.customersRepository.findOne({ where: { genCustomerId: updateCustomerDto.gen_customer_id } });
      if (existingCustomer) {
        throw new ConflictException('Generated customer ID already exists');
      }
    }

    if (updateCustomerDto.assigned_to_id) {
      const assignee = await this.usersRepository.findOne({ where: { id: parseInt(updateCustomerDto.assigned_to_id.toString(), 10) } });
      if (!assignee) {
        throw new NotFoundException(`User with ID ${updateCustomerDto.assigned_to_id} not found`);
      }
    }

    const updatedCustomer = this.customersRepository.merge(customer, {
      firstName: updateCustomerDto.first_name,
      lastName: updateCustomerDto.last_name,
      email: updateCustomerDto.email,
      mobileNumber: updateCustomerDto.mobile_phone,
      landline: updateCustomerDto.home_phone,
      title: updateCustomerDto.title,
      iso: updateCustomerDto.iso,
      dnc: updateCustomerDto.DNC,
      isLandLineDefault: updateCustomerDto.is_land_line_default,
      isActive: updateCustomerDto.is_active,
      customerId: updateCustomerDto.customer_id,
      genCustomerId: updateCustomerDto.gen_customer_id,
      assigned_to_id: updateCustomerDto.assigned_to_id ? parseInt(updateCustomerDto.assigned_to_id.toString(), 10) : customer.assigned_to_id,
    });

    const savedCustomer = await this.customersRepository.save(updatedCustomer);

    if (updateCustomerDto.address) {
      const existingAddresses = customer.addresses || [];
      const updatedAddressIds = updateCustomerDto.address
        .filter((addr) => addr.id)
        .map((addr) => addr.id);

      for (const existingAddress of existingAddresses) {
        if (!updatedAddressIds.includes(existingAddress.id)) {
          existingAddress.isActive = false;
          await this.addressesRepository.save(existingAddress);
        }
      }

      const addresses = await Promise.all(
        updateCustomerDto.address.map(async (addr) => {
          if (addr.id) {
            const existingAddress = existingAddresses.find((ea) => ea.id === addr.id);
            if (existingAddress) {
              return this.addressesRepository.merge(existingAddress, {
                ...addr,
                isActive: addr.is_active !== undefined ? addr.is_active : existingAddress.isActive,
              });
            }
          }
          return this.addressesRepository.create({
            ...addr,
            customer: savedCustomer,
            isActive: addr.is_active !== undefined ? addr.is_active : true,
          });
        }),
      );

      await this.addressesRepository.save(addresses);
      savedCustomer.addresses = addresses;
    }

    return savedCustomer;
  }

  async updateById(id: number, updateCustomerDto: UpdateCustomerDto, user: any): Promise<Customer> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    console.log(`Updating customer with id: ${id}`);

    const customer = await this.customersRepository.findOne({ where: { id }, relations: ['addresses'] });
    if (!customer) {
      throw new NotFoundException(`Customer with id ${id} not found`);
    }

    const roleName = typeof user.role === 'string' ? user.role : user.role?.name || '';
    if (customer.assigned_to_id !== user.id && customer.createdById !== user.id && roleName.toLowerCase() !== 'admin' && roleName.toLowerCase() !== 'owner') {
      throw new UnauthorizedException('Not authorized to update this customer');
    }

    if (updateCustomerDto.email && updateCustomerDto.email !== customer.email) {
      const existingCustomer = await this.customersRepository.findOne({ where: { email: updateCustomerDto.email } });
      if (existingCustomer) {
        throw new ConflictException('Email already exists');
      }
    }

    if (updateCustomerDto.gen_customer_id && updateCustomerDto.gen_customer_id !== customer.genCustomerId) {
      const existingCustomer = await this.customersRepository.findOne({ where: { genCustomerId: updateCustomerDto.gen_customer_id } });
      if (existingCustomer) {
        throw new ConflictException('Generated customer ID already exists');
      }
    }

    if (updateCustomerDto.assigned_to_id) {
      const assignee = await this.usersRepository.findOne({ where: { id: parseInt(updateCustomerDto.assigned_to_id.toString(), 10) } });
      if (!assignee) {
        throw new NotFoundException(`User with ID ${updateCustomerDto.assigned_to_id} not found`);
      }
    }

    const updatedCustomer = this.customersRepository.merge(customer, {
      firstName: updateCustomerDto.first_name,
      lastName: updateCustomerDto.last_name,
      email: updateCustomerDto.email,
      mobileNumber: updateCustomerDto.mobile_phone,
      landline: updateCustomerDto.home_phone,
      title: updateCustomerDto.title,
      iso: updateCustomerDto.iso,
      dnc: updateCustomerDto.DNC,
      isLandLineDefault: updateCustomerDto.is_land_line_default,
      isActive: updateCustomerDto.is_active,
      customerId: updateCustomerDto.customer_id,
      genCustomerId: updateCustomerDto.gen_customer_id,
      assigned_to_id: updateCustomerDto.assigned_to_id ? parseInt(updateCustomerDto.assigned_to_id.toString(), 10) : customer.assigned_to_id,
    });

    if (updateCustomerDto.address) {
      const existingAddresses = customer.addresses || [];
      const updatedAddressIds = updateCustomerDto.address
        .filter((addr) => addr.id)
        .map((addr) => addr.id);

      for (const existingAddress of existingAddresses) {
        if (!updatedAddressIds.includes(existingAddress.id)) {
          existingAddress.isActive = false;
          await this.addressesRepository.save(existingAddress);
        }
      }

      const addresses = await Promise.all(
        updateCustomerDto.address.map(async (addr) => {
          if (addr.id) {
            const existingAddress = existingAddresses.find((ea) => ea.id === addr.id);
            if (existingAddress) {
              return this.addressesRepository.merge(existingAddress, {
                ...addr,
                isActive: addr.is_active !== undefined ? addr.is_active : existingAddress.isActive,
              });
            }
          }
          return this.addressesRepository.create({
            ...addr,
            customer: updatedCustomer,
            isActive: addr.is_active !== undefined ? addr.is_active : true,
          });
        }),
      );
      updatedCustomer.addresses = addresses;
    }

    return this.customersRepository.save(updatedCustomer);
  }

  async findOne(id: number, user: any): Promise<any> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    const roleName =
  typeof user.role === 'string'
    ? user.role
    : user.role?.name || '';


    const customer = await this.customersRepository.findOne({
      where: { id },
      relations: ['addresses', 'assignedTo', 'assignedTo.reportsTo'],
    });
    if (!customer) {
      throw new NotFoundException(`Customer with ID ${id} not found`);
    }

    // Filter only active addresses
    const activeAddresses = customer.addresses?.filter(addr => addr.isActive !== false) || [];

    // const roleName = typeof user.role === 'string' ? user.role : user.role || '';
    // if (customer.assigned_to_id !== user.id && customer.createdById !== user.id && roleName.toLowerCase() !== 'admin' && roleName.toLowerCase() !== 'owner') {
    //   throw new UnauthorizedException('Not authorized to view this customer');
    // }

    const notes = await this.customerNotesRepository.find({
      where: { customer: { id } },
      relations: ['parentNote'],
      order: { createdAt: 'DESC' },
    });

    const creator = await this.usersRepository.findOne({ where: { id: customer.createdById } });

    const transformedNotes = notes.map(note => ({
      id: note.id,
      replies: [],
      created_by: {
        id: user.id,
        user: {
          email: user.email,
          name: creator ? `${creator.firstName || ''} ${creator.lastName || ''}`.trim() || user.email : user.email,
          last_login: user.lastLogin || new Date(),
        },
        granted_permissions: user.permissions || [],
        role: {
          id: '',
          name: roleName || '',
          permissions: [],
        },
        reports_to: user.reportsTo ? user.reportsTo.id : null,
        created_at: user.createdAt || new Date(),
        updated_at: user.updatedAt || new Date(),
        is_active: user.isActive,
      },
      total_parents_count: 0,
      created_at: note.createdAt,
      updated_at: note.updatedAt,
      is_active: true,
      note: note.note,
      customer: customer.id,
      parent_note: note.parentNote ? note.parentNote.id : null,
      child_note: null,
    }));

    return {
      id: customer.id,
      customer_notes: transformedNotes,
      address: activeAddresses,
      assigned_to: customer.assignedTo ? {
        id: customer.assignedTo.id,
        user: {
          email: customer.assignedTo.email || '',
          name: `${customer.assignedTo.firstName || ''} ${customer.assignedTo.lastName || ''}`.trim() || customer.assignedTo.email || '',
          last_login: customer.assignedTo.lastLogin || null,
        },
        granted_permissions: customer.assignedTo.permissions || [],
        role: {
          id: '',
          name: customer.assignedTo.role || '',
          permissions: [],
        },
        reports_to: customer.assignedTo.reportsTo ? customer.assignedTo.reportsTo.id : null,
        created_at: customer.assignedTo.createdAt || null,
        updated_at: customer.assignedTo.updatedAt || null,
        is_active: customer.assignedTo.isActive ?? true,
      } : null,
      created_by: {
        id: user.id,
        user: {
          email: user.email,
          name: creator ? `${creator.firstName || ''} ${creator.lastName || ''}`.trim() || user.email : user.email,
          last_login: user.lastLogin || new Date(),
        },
        granted_permissions: user.permissions || [],
        role: {
          id: '',
          name: roleName || '',
          permissions: [],
        },
        reports_to: user.reportsTo ? user.reportsTo.id : null,
        created_at: user.createdAt || new Date(),
        updated_at: user.updatedAt || new Date(),
        is_active: user.isActive,
      },
      created_at: customer.createdAt,
      updated_at: customer.updatedAt,
      is_active: customer.isActive,
      customer_id: customer.customerId,
      gen_customer_id: customer.genCustomerId,
      title: customer.title,
      first_name: customer.firstName,
      last_name: customer.lastName,
      email: customer.email,
      mobile_phone: customer.dnc === true ? 'DNC' : customer.mobileNumber || '',
      home_phone: customer.dnc === true ? 'DNC' : customer.landline || '',
      is_land_line_default: customer.isLandLineDefault,
      iso: customer.iso,
      DNC: customer.dnc,
    };
  }

  async findOneByCustomerId(customerId: string, user: any): Promise<any> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    const roleName =
    typeof user.role === 'string'
    ? user.role
    : user.role?.name || '';


    console.log(`Querying customer with customer_id: ${customerId} (type: ${typeof customerId}) for user ID: ${user.id}`);

    const customer = await this.customersRepository.findOne({
      where: { customerId: customerId.toString() },
      relations: ['addresses', 'assignedTo', 'assignedTo.reportsTo'],
    });

    if (!customer) {
      console.error(`No customer found with customer_id: ${customerId}`);
      throw new NotFoundException(`Customer with customer_id ${customerId} not found`);
    }

    // Filter only active addresses
    const activeAddresses = customer.addresses?.filter(addr => addr.isActive !== false) || [];

    // const roleName = typeof user.role === 'string' ? user.role : user.role?.name || '';
    // console.log(`User role: ${roleName}, assigned_to_id: ${customer.assigned_to_id}, createdById: ${customer.createdById}, user.id: ${user.id}`);
    // if (customer.assigned_to_id !== user.id && customer.createdById !== user.id && roleName.toLowerCase() !== 'admin' && roleName.toLowerCase() !== 'owner') {
    //   console.error(`User ${user.id} not authorized to view customer ${customerId}`);
    //   throw new UnauthorizedException('Not authorized to view this customer');
    // }

    const notes = await this.customerNotesRepository.find({
      where: { customer: { id: customer.id } },
      relations: ['parentNote'],
      order: { createdAt: 'DESC' },
    });

    const creator = await this.usersRepository.findOne({ where: { id: customer.createdById } });

    const transformedNotes = notes.map(note => ({
      id: note.id,
      replies: [],
      created_by: {
        id: user.id,
        user: {
          email: user.email,
          name: creator ? `${creator.firstName || ''} ${creator.lastName || ''}`.trim() || user.email : user.email,
          last_login: user.lastLogin || new Date(),
        },
        granted_permissions: user.permissions || [],
        role: {
          id: '',
          name: roleName || '',
          permissions: [],
        },
        reports_to: user.reportsTo ? user.reportsTo.id : null,
        created_at: user.createdAt || new Date(),
        updated_at: user.updatedAt || new Date(),
        is_active: user.isActive,
      },
      total_parents_count: 0,
      created_at: note.createdAt,
      updated_at: note.updatedAt,
      is_active: true,
      note: note.note,
      customer: customer.id,
      parent_note: note.parentNote ? note.parentNote.id : null,
      child_note: null,
    }));

    return {
      id: customer.id,
      customer_notes: transformedNotes,
      address: (customer.addresses || []).filter(addr => addr.isActive !== false),
      assigned_to: customer.assignedTo ? {
        id: customer.assignedTo.id,
        user: {
          email: customer.assignedTo.email || '',
          name: `${customer.assignedTo.firstName || ''} ${customer.assignedTo.lastName || ''}`.trim() || customer.assignedTo.email || '',
          last_login: customer.assignedTo.lastLogin || null,
        },
        granted_permissions: customer.assignedTo.permissions || [],
        role: {
          id: '',
          name: customer.assignedTo.role || '',
          permissions: [],
        },
        reports_to: customer.assignedTo.reportsTo ? customer.assignedTo.reportsTo.id : null,
        created_at: customer.assignedTo.createdAt || null,
        updated_at: customer.assignedTo.updatedAt || null,
        is_active: customer.assignedTo.isActive ?? true,
      } : null,
      created_by: {
        id: user.id,
        user: {
          email: user.email,
          name: creator ? `${creator.firstName || ''} ${creator.lastName || ''}`.trim() || user.email : user.email,
          last_login: user.lastLogin || new Date(),
        },
        granted_permissions: user.permissions || [],
        role: {
          id: '',
          name: roleName || '',
          permissions: [],
        },
        reports_to: user.reportsTo ? user.reportsTo.id : null,
        created_at: user.createdAt || new Date(),
        updated_at: user.updatedAt || new Date(),
        is_active: user.isActive,
      },
      created_at: customer.createdAt,
      updated_at: customer.updatedAt,
      is_active: customer.isActive,
      customer_id: customer.customerId,
      gen_customer_id: customer.genCustomerId,
      title: customer.title,
      first_name: customer.firstName,
      last_name: customer.lastName,
      email: customer.email,
      mobile_phone: customer.dnc === true ? 'DNC' : customer.mobileNumber || '',
      home_phone: customer.dnc === true ? 'DNC' : customer.landline || '',
      is_land_line_default: customer.isLandLineDefault,
      iso: customer.iso,
      DNC: customer.dnc,
    };
  }

  async findAll(listCustomersDto: ListCustomersDto, user: any): Promise<{ data: any[]; total: number }> {
    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    const roleName =
    typeof user.role === 'string'
    ? user.role
    : user.role?.name || '';


    const { offset = 0, limit = 10, search, scope } = listCustomersDto;
    // const roleName = typeof user.role === 'string' ? user.role : user.role?.name || '';

    // const qb = this.customersRepository.createQueryBuilder('customer')
    //   .leftJoinAndSelect('customer.addresses', 'addresses')
    //   .leftJoinAndSelect('customer.assignedTo', 'assignedTo')
    //   .leftJoinAndSelect('assignedTo.reportsTo', 'reportsTo')
    //   .skip(offset)
    //   .take(limit)
    //   .orderBy('customer.createdAt', 'DESC');

    // if (roleName.toLowerCase() !== 'admin' && roleName.toLowerCase() !== 'owner') {
    //   qb.andWhere('(customer.assigned_to_id = :userId OR customer.createdById = :userId)', { userId: user.id });
    // }

    const qb = this.customersRepository.createQueryBuilder('customer')
      .leftJoinAndSelect('customer.addresses', 'addresses')
      .leftJoinAndSelect('customer.assignedTo', 'assignedTo')
      .leftJoinAndSelect('assignedTo.reportsTo', 'reportsTo')
      .skip(offset)
      .take(limit)
      .orderBy('customer.createdAt', 'DESC');

    const fieldMap: Record<string, string> = {
      first_name: 'customer.firstName',
      last_name: 'customer.lastName',
      email: 'customer.email',
      mobile_phone: 'customer.mobileNumber',
      home_phone: 'customer.landline',
      id: 'customer.id',
    };

    if (scope === 'created_by') {
      qb.andWhere('customer.createdById = :createdById', { createdById: user.id });
    } else if (scope && !fieldMap[scope]) {
      qb.andWhere('customer.iso = :iso', { iso: scope });
    }

    if (search) {
      if (scope && fieldMap[scope]) {
        qb.andWhere(`${fieldMap[scope]} LIKE :search`, { search: `%${search}%` });
      } else {
        qb.andWhere(
          [
            'customer.firstName LIKE :search',
            'customer.lastName LIKE :search',
            'customer.id LIKE :search',
            'customer.email LIKE :search',
            'customer.mobileNumber LIKE :search',
            'customer.landline LIKE :search',
          ].join(' OR '),
          { search: `%${search}%` }
        );
      }
    }

    const [data, total] = await qb.getManyAndCount();

    const creatorIds = [...new Set(data.map(customer => customer.createdById))];
    const creators = await this.usersRepository.find({ where: { id: In(creatorIds) } });
    const creatorMap = new Map(creators.map(creator => [creator.id, creator]));

    const transformedData = data.map(customer => ({
      id: customer.id,
      customer_notes: [], // Required by frontend CustomerDataT interface
      first_name: customer.firstName,
      last_name: customer.lastName,
      email: customer.email,
      mobile_phone: customer.dnc === true ? 'DNC' : customer.mobileNumber || '',
      home_phone: customer.dnc === true ? 'DNC' : customer.landline || '',
      address: (customer.addresses || []).filter(addr => addr.isActive !== false),
      title: customer.title,
      iso: customer.iso,
      DNC: customer.dnc,
      is_land_line_default: customer.isLandLineDefault,
      created_at: customer.createdAt,
      updated_at: customer.updatedAt, // Required by frontend
      is_active: customer.isActive ?? true, // Required by frontend
      customer_id: customer.customerId,
      gen_customer_id: customer.genCustomerId,
      assigned_to: customer.assignedTo ? {
        id: customer.assignedTo.id,
        user: {
          email: customer.assignedTo.email || '',
          name: `${customer.assignedTo.firstName || ''} ${customer.assignedTo.lastName || ''}`.trim() || customer.assignedTo.email || '',
          last_login: customer.assignedTo.lastLogin || null,
        },
        granted_permissions: customer.assignedTo.permissions || [],
        role: {
          id: '',
          name: customer.assignedTo.role || '',
          permissions: [],
        },
        reports_to: customer.assignedTo.reportsTo ? customer.assignedTo.reportsTo.id : null,
        created_at: customer.assignedTo.createdAt || null,
        updated_at: customer.assignedTo.updatedAt || null,
        is_active: customer.assignedTo.isActive ?? true,
      } : null,
      created_by: {
        id: customer.createdById,
        user: {
          email: creatorMap.get(customer.createdById)?.email || user.email,
          name: creatorMap.get(customer.createdById) ? 
            `${creatorMap.get(customer.createdById)?.firstName || ''} ${creatorMap.get(customer.createdById)?.lastName || ''}`.trim() || user.email : user.email,
          last_login: creatorMap.get(customer.createdById)?.lastLogin || user.lastLogin || new Date(),
        },
        granted_permissions: user.permissions || [],
        role: {
          id: '',
          name: roleName || '',
          permissions: [],
        },
        reports_to: user.reportsTo ? user.reportsTo.id : null,
        created_at: creatorMap.get(customer.createdById)?.createdAt || user.createdAt || new Date(),
        updated_at: creatorMap.get(customer.createdById)?.updatedAt || user.updatedAt || new Date(),
        is_active: creatorMap.get(customer.createdById)?.isActive ?? user.isActive,
      },
    }));

    return { data: transformedData, total };
  }

  async bulkAssign(dto: BulkAssignCustomerDto, user: any): Promise<string> {
    const { customer_ids, assigned_to_id } = dto;

    if (!user || !user.id) {
      throw new UnauthorizedException('User must be authenticated');
    }

    const assignee = await this.usersRepository.findOne({ where: { id: assigned_to_id } });
    if (!assignee) {
      throw new NotFoundException(`User with ID ${assigned_to_id} not found`);
    }

    const customers = await this.customersRepository.findBy({ id: In(customer_ids) });

    if (!customers.length) {
      throw new NotFoundException(`No customers found with the provided IDs`);
    }

    for (const customer of customers) {
      customer.assigned_to_id = assigned_to_id;
    }

    await this.customersRepository.save(customers);

    return `Selected customers have been assigned to ${assignee.firstName || ''}_${assignee.lastName || 'user'}`;
  }

  async bulkCreate(file: Express.Multer.File, user: any): Promise<{
    message: string;
    details: { success: number; failed: number; errors: string[]; duplicateIds: string[] };
  }> {
    const results = { success: 0, failed: 0, errors: [] as string[], duplicateIds: [] as string[] };
  
    try {
      // Validate file exists
      if (!file || !file.buffer) {
        throw new BadRequestException('File is required and must not be empty');
      }

      // Read Excel file
      let workbook;
      try {
        workbook = XLSX.read(file.buffer, { type: 'buffer' });
      } catch (error) {
        throw new BadRequestException(`Failed to read Excel file: ${error.message}. Please ensure the file is a valid Excel format (.xlsx or .xls)`);
      }

      // Validate workbook has sheets
      if (!workbook.SheetNames || workbook.SheetNames.length === 0) {
        throw new BadRequestException('Excel file has no sheets. Please ensure the file contains at least one worksheet.');
      }

      const worksheet = workbook.Sheets[workbook.SheetNames[0]];
      
      // Validate worksheet exists
      if (!worksheet) {
        throw new BadRequestException('Excel file worksheet is empty or corrupted. Please check the file and try again.');
      }

      // Convert worksheet to JSON
      let records: ExcelCustomerRecord[];
      try {
        records = XLSX.utils.sheet_to_json<ExcelCustomerRecord>(worksheet);
      } catch (error) {
        throw new BadRequestException(`Failed to parse Excel data: ${error.message}. Please check the file format.`);
      }

      // Validate records array exists and is not undefined
      if (!records || !Array.isArray(records)) {
        throw new BadRequestException('Excel file could not be parsed. Please ensure the file is a valid Excel format and contains data.');
      }
  
      // Validate file is not empty
      if (records.length === 0) {
        throw new BadRequestException('The Excel file is empty. Please ensure the file contains at least one data row (in addition to the header row).');
      }
  
      // Validate required columns - Check if records[0] exists before accessing
      if (!records[0] || typeof records[0] !== 'object') {
        throw new BadRequestException('Excel file format is invalid. Please ensure the first row contains headers and the second row contains data.');
      }

      const requiredColumns = ['first_name', 'last_name'];
      const missingColumns = requiredColumns.filter(col => !(col in records[0]));
      if (missingColumns.length > 0) {
        throw new BadRequestException(`Missing required columns: ${missingColumns.join(', ')}. Please ensure your Excel file has these columns in the first row.`);
      }
  
      // Step 1: Check for duplicates within the Excel file itself
      // Instead of throwing error, we'll filter them out and continue
      const duplicateRowsInFile = this.findDuplicateRowsInFile(records);
  
      // Step 2: Check for existing records in database (batch check for performance)
      // We'll filter these out and continue with the rest
      const existingCustomerIds = new Set<string>();
      const existingMobileNumbers = new Set<string>();
      const existingHomePhones = new Set<string>();
      const existingEmails = new Set<string>();
      
      // Collect all customer IDs, mobile numbers, home phones, and emails from Excel
      const customerIds: string[] = [];
      const mobileNumbers: string[] = [];
      const homePhones: string[] = [];
      const emails: string[] = [];
      
      records.forEach((record) => {
        if (record.id) {
          const id = record.id.toString().trim();
          if (id) customerIds.push(id);
        }
        if (record.mobile_phone) {
          const mobile = record.mobile_phone.toString().trim();
          if (mobile) mobileNumbers.push(mobile);
        }
        if (record.home_phone) {
          const homePhone = record.home_phone.toString().trim();
          if (homePhone) homePhones.push(homePhone);
        }
        if (record.email) {
          const email = record.email.toString().trim().toLowerCase();
          if (email) emails.push(email);
        }
      });
      
      // Batch check for existing customer IDs
      if (customerIds.length > 0) {
        const existingCustomersById = await this.customersRepository.find({
          where: { genCustomerId: In(customerIds) },
          select: ['id', 'genCustomerId'],
        });
        existingCustomersById.forEach((customer) => {
          if (customer.genCustomerId) {
            existingCustomerIds.add(customer.genCustomerId);
          }
        });
      }
      
      // Batch check for existing mobile numbers
      if (mobileNumbers.length > 0) {
        const existingCustomersByMobile = await this.customersRepository.find({
          where: { mobileNumber: In(mobileNumbers) },
          select: ['id', 'mobileNumber'],
        });
        existingCustomersByMobile.forEach((customer) => {
          if (customer.mobileNumber) {
            existingMobileNumbers.add(customer.mobileNumber);
          }
        });
      }
      
      // Batch check for existing home phones
      if (homePhones.length > 0) {
        const existingCustomersByHomePhone = await this.customersRepository.find({
          where: { landline: In(homePhones) },
          select: ['id', 'landline'],
        });
        existingCustomersByHomePhone.forEach((customer) => {
          if (customer.landline) {
            existingHomePhones.add(customer.landline);
          }
        });
      }
      
      // Batch check for existing emails
      if (emails.length > 0) {
        const existingCustomersByEmail = await this.customersRepository.find({
          where: { email: In(emails) },
          select: ['id', 'email'],
        });
        existingCustomersByEmail.forEach((customer) => {
          if (customer.email) {
            existingEmails.add(customer.email.toLowerCase());
          }
        });
      }
  
      // Step 3: Process each record
      for (let index = 0; index < records.length; index++) {
        const record = records[index];
        const rowNumber = index + 2; // +2 because Excel rows start at 1, and row 1 is header
  
        try {
          // Skip if this row is a duplicate within the Excel file itself
          const duplicateReasonsInFile = duplicateRowsInFile.get(index);
          if (duplicateReasonsInFile && duplicateReasonsInFile.length > 0) {
            results.failed++;
            const recordId = record.id?.toString().trim();
            if (recordId) {
              results.duplicateIds.push(recordId);
            }
            results.errors.push(`Row ${rowNumber}: Duplicate found within Excel file - ${duplicateReasonsInFile.join(', ')} already appears in a previous row`);
            continue;
          }
          
          // Check if this record is a duplicate in database (skip it and continue)
          const recordId = record.id?.toString().trim();
          const recordMobile = record.mobile_phone?.toString().trim();
          const recordHomePhone = record.home_phone?.toString().trim();
          const recordEmail = record.email?.toString().trim().toLowerCase();
          
          let isDuplicate = false;
          const duplicateReasons: string[] = [];
          
          // Check customer ID
          if (recordId && existingCustomerIds.has(recordId)) {
            duplicateReasons.push(`Customer ID "${recordId}"`);
            isDuplicate = true;
          }
          
          // Check mobile phone
          if (recordMobile && existingMobileNumbers.has(recordMobile)) {
            duplicateReasons.push(`Mobile number "${recordMobile}"`);
            isDuplicate = true;
          }
          
          // Check home phone
          if (recordHomePhone && existingHomePhones.has(recordHomePhone)) {
            duplicateReasons.push(`Home phone "${recordHomePhone}"`);
            isDuplicate = true;
          }
          
          // Check email
          if (recordEmail && existingEmails.has(recordEmail)) {
            duplicateReasons.push(`Email "${recordEmail}"`);
            isDuplicate = true;
          }
          
          if (isDuplicate) {
            results.failed++;
            if (recordId) {
              results.duplicateIds.push(recordId);
            }
            results.errors.push(`Row ${rowNumber}: Duplicate found - ${duplicateReasons.join(', ')} already exists in database`);
            continue; // Skip this record and continue with the next one
          }
  
          // Validate required fields
          if (!record.first_name || !record.last_name) {
            throw new BadRequestException('First name and last name are required');
          }
  
          // Validate mobile number if provided
          if (record.mobile_phone) {
            const trimmedMobile = record.mobile_phone.toString().trim();
            if (trimmedMobile.length === 0) {
              throw new BadRequestException('Mobile phone cannot be empty if provided');
            }
          }
  
          // Validate customer ID if provided
          if (record.id) {
            const trimmedId = record.id.toString().trim();
            if (trimmedId.length === 0) {
              throw new BadRequestException('Customer ID cannot be empty if provided');
            }
          }
  
          // Extract address data from Excel record
          const addressFields = {
            address1: record.address1?.toString().trim() || undefined,
            address2: record.address2?.toString().trim() || undefined,
            town_city: record.town_city?.toString().trim() || undefined,
            county: record.county?.toString().trim() || undefined,
            country: record.country?.toString().trim() || undefined,
            post_code: record.post_code?.toString().trim() || undefined,
            formatted_address: record.formatted_address?.toString().trim() || undefined,
            primary: record.primary !== undefined 
              ? (typeof record.primary === 'boolean' 
                  ? record.primary 
                  : record.primary.toString().toLowerCase() === 'true' || record.primary.toString() === '1')
              : false,
            hasAtLeastOneField: true, // Required by AddressDto validation
          };

          // Check if at least one address field has a value (excluding the validation marker)
          const hasAddressData = (
            !!addressFields.address1 ||
            !!addressFields.address2 ||
            !!addressFields.town_city ||
            !!addressFields.county ||
            !!addressFields.country ||
            !!addressFields.post_code ||
            !!addressFields.formatted_address ||
            addressFields.primary === true
          );

          // Create address array if address data exists
          const addressArray = hasAddressData ? [addressFields] : undefined;

          // Create DTO from Excel record
          const createCustomerDto: CreateCustomerDto = {
            first_name: record.first_name?.trim() || '',
            last_name: record.last_name?.trim() || '',
            email: record.email?.trim() || undefined,
            mobile_phone: record.mobile_phone?.toString().trim() || undefined,
            home_phone: record.home_phone?.toString().trim() || undefined,
            title: record.title?.trim() || undefined,
            iso: record.iso?.trim() || undefined,
            assigned_to_id: record.assigned_to_id 
              ? parseInt(record.assigned_to_id.toString(), 10) 
              : undefined,
            address: addressArray,
          };
  
          // Create customer using existing create method
          const customer = await this.create(createCustomerDto, user);
  
          // If Excel has a customer ID, update the genCustomerId
          if (record.id) {
            const customerId = record.id.toString().trim();
            
            // Double-check uniqueness before updating
            const existingWithId = await this.customersRepository.findOne({
              where: { genCustomerId: customerId },
            });
  
            if (existingWithId && existingWithId.id !== customer.id) {
              // This shouldn't happen, but handle it gracefully
              results.failed++;
              results.duplicateIds.push(customerId);
              results.errors.push(`Row ${rowNumber}: Customer ID "${customerId}" already exists`);
              continue;
            }
  
            customer.genCustomerId = customerId;
            await this.customersRepository.save(customer);
          }
  
          results.success++;
        } catch (error) {
          results.failed++;
          const errorMessage = error.message || 'Unknown error';
          const recordId = record.id?.toString().trim();
          if (recordId && !results.duplicateIds.includes(recordId)) {
            // Only add to duplicateIds if it's a conflict/duplicate error
            if (errorMessage.includes('already exists') || errorMessage.includes('duplicate')) {
              results.duplicateIds.push(recordId);
            }
          }
          results.errors.push(`Row ${rowNumber}: ${errorMessage}`);
        }
      }
  
      // Generate response message
      let message = '';
      if (results.success > 0 && results.failed === 0) {
        message = `Successfully created ${results.success} customer${results.success > 1 ? 's' : ''}.`;
      } else if (results.success > 0 && results.failed > 0) {
        message = `Successfully created ${results.success} customer${results.success > 1 ? 's' : ''}. ${results.failed} record${results.failed > 1 ? 's' : ''} skipped due to duplicates.`;
      } else {
        message = `No customers were created. All records were duplicates or had errors.`;
      }
  
      return { message, details: results };
    } catch (error) {
      // If it's a validation error from our checks, throw it as-is
      if (error instanceof BadRequestException || error instanceof ConflictException) {
        throw error;
      }
      // Otherwise, wrap in BadRequestException
      throw new BadRequestException(error.message || 'Invalid Excel file');
    }
  }
  
  /**
   * Finds duplicate rows within the Excel file itself
   * Checks for duplicate customer IDs, mobile numbers, home phones, and emails
   * Returns a Map of row indices (0-based) to duplicate reasons that should be skipped (keeping first occurrence)
   */
  private findDuplicateRowsInFile(records: ExcelCustomerRecord[]): Map<number, string[]> {
    const duplicateRows = new Map<number, string[]>(); // Map<rowIndex, reasons[]>
    const seenIds = new Map<string, number>(); // Map<id, firstRowIndex>
    const seenMobileNumbers = new Map<string, number>(); // Map<mobileNumber, firstRowIndex>
    const seenHomePhones = new Map<string, number>(); // Map<homePhone, firstRowIndex>
    const seenEmails = new Map<string, number>(); // Map<email, firstRowIndex>
  
    records.forEach((record, index) => {
      const reasons: string[] = [];
      
      // Check customer ID duplicates
      if (record.id) {
        const id = record.id.toString().trim().toLowerCase();
        if (id) {
          if (seenIds.has(id)) {
            reasons.push(`Customer ID "${id}"`);
          } else {
            seenIds.set(id, index);
          }
        }
      }
  
      // Check mobile number duplicates
      if (record.mobile_phone) {
        const mobileNumber = record.mobile_phone.toString().trim();
        if (mobileNumber) {
          if (seenMobileNumbers.has(mobileNumber)) {
            reasons.push(`Mobile number "${mobileNumber}"`);
          } else {
            seenMobileNumbers.set(mobileNumber, index);
          }
        }
      }
  
      // Check home phone duplicates
      if (record.home_phone) {
        const homePhone = record.home_phone.toString().trim();
        if (homePhone) {
          if (seenHomePhones.has(homePhone)) {
            reasons.push(`Home phone "${homePhone}"`);
          } else {
            seenHomePhones.set(homePhone, index);
          }
        }
      }
  
      // Check email duplicates
      if (record.email) {
        const email = record.email.toString().trim().toLowerCase();
        if (email) {
          if (seenEmails.has(email)) {
            reasons.push(`Email "${email}"`);
          } else {
            seenEmails.set(email, index);
          }
        }
      }
      
      // If any duplicates found, mark this row
      if (reasons.length > 0) {
        duplicateRows.set(index, reasons);
      }
    });
  
    return duplicateRows;
  }
  
  /**
   * Validates for duplicates within the Excel file itself (legacy method, kept for compatibility)
   * Checks for duplicate customer IDs and mobile numbers
   */
  private validateDuplicatesInFile(records: ExcelCustomerRecord[]): {
    hasDuplicates: boolean;
    errors: string[];
  } {
    const errors: string[] = [];
    const seenIds = new Map<string, number[]>(); // Map<id, rowNumbers[]>
    const seenMobileNumbers = new Map<string, number[]>(); // Map<mobileNumber, rowNumbers[]>
  
    records.forEach((record, index) => {
      const rowNumber = index + 2; // +2 because Excel rows start at 1, and row 1 is header
  
      // Check customer ID duplicates
      if (record.id) {
        const id = record.id.toString().trim().toLowerCase();
        if (id) {
          if (seenIds.has(id)) {
            const existingRows = seenIds.get(id) || [];
            existingRows.push(rowNumber);
            seenIds.set(id, existingRows);
            errors.push(`Customer ID "${id}" appears in rows: ${existingRows.join(', ')}`);
          } else {
            seenIds.set(id, [rowNumber]);
          }
        }
      }
  
      // Check mobile number duplicates
      if (record.mobile_phone) {
        const mobileNumber = record.mobile_phone.toString().trim();
        if (mobileNumber) {
          if (seenMobileNumbers.has(mobileNumber)) {
            const existingRows = seenMobileNumbers.get(mobileNumber) || [];
            existingRows.push(rowNumber);
            seenMobileNumbers.set(mobileNumber, existingRows);
            errors.push(`Mobile number "${mobileNumber}" appears in rows: ${existingRows.join(', ')}`);
          } else {
            seenMobileNumbers.set(mobileNumber, [rowNumber]);
          }
        }
      }
    });
  
    return {
      hasDuplicates: errors.length > 0,
      errors,
    };
  }
  
  /**
   * Checks for existing records in the database
   * Validates customer IDs and mobile numbers against existing records
   */
  private async checkExistingRecords(records: ExcelCustomerRecord[]): Promise<{
    hasExisting: boolean;
    errors: string[];
  }> {
    const errors: string[] = [];
    const customerIds: string[] = [];
    const mobileNumbers: string[] = [];
  
    // Collect all customer IDs and mobile numbers from Excel
    records.forEach((record, index) => {
      const rowNumber = index + 2;
  
      if (record.id) {
        const id = record.id.toString().trim();
        if (id) {
          customerIds.push(id);
        }
      }
  
      if (record.mobile_phone) {
        const mobile = record.mobile_phone.toString().trim();
        if (mobile) {
          mobileNumbers.push(mobile);
        }
      }
    });
  
    // Check for existing customer IDs
    if (customerIds.length > 0) {
      const existingCustomersById = await this.customersRepository.find({
        where: { genCustomerId: In(customerIds) },
        select: ['id', 'genCustomerId'],
      });
  
      if (existingCustomersById.length > 0) {
        existingCustomersById.forEach((customer) => {
          errors.push(`Customer ID "${customer.genCustomerId}" already exists in database`);
        });
      }
    }
  
    // Check for existing mobile numbers
    if (mobileNumbers.length > 0) {
      const existingCustomersByMobile = await this.customersRepository.find({
        where: { mobileNumber: In(mobileNumbers) },
        select: ['id', 'mobileNumber'],
      });
  
      if (existingCustomersByMobile.length > 0) {
        existingCustomersByMobile.forEach((customer) => {
          errors.push(`Mobile number "${customer.mobileNumber}" already exists in database`);
        });
      }
    }
  
    return {
      hasExisting: errors.length > 0,
      errors,
    };
  }
}