import { Injectable, CanActivate, ExecutionContext, ForbiddenException, Logger } from '@nestjs/common';
import { Reflector } from '@nestjs/core';
import { Request } from 'express';

@Injectable()
export class PermissionsGuard implements CanActivate {
  private readonly logger = new Logger(PermissionsGuard.name);

  constructor(private reflector: Reflector) {}

  canActivate(context: ExecutionContext): boolean {
    // Get the required permission from metadata (set by @RequirePermission decorator)
    const requiredPermission = this.reflector.get<string>('permission', context.getHandler());
    if (!requiredPermission) {
      // If no permission is specified, allow the request (fall back to other guards)
      return true;
    }

    const request = context.switchToHttp().getRequest<Request>();
    const user = request.user as any; // User object from JWT payload

    this.logger.debug(`Required permission: ${requiredPermission}`);
    this.logger.debug(`User role: ${user?.role}`);
    this.logger.debug(`User permissions: ${JSON.stringify(user?.permissions)}`);

    // If user is Admin or Owner, bypass permission check
    if (user?.role === 'Admin' || user?.role === 'Owner') {
      this.logger.debug('Bypassing permission check for Admin/Owner user');
      return true;
    }

    if (!user || !user.permissions) {
      this.logger.error('User or permissions not found in token');
      throw new ForbiddenException('User permissions not found in token');
    }

    // Check if the user has the required permission (by name or codename)
    const requiredCodename = requiredPermission
      .toLowerCase()
      .replace(/\s+/g, '_')
      .replace(/^can_/, ''); // Remove leading 'can_' if present, to match codename style
    let hasPermission = user.permissions.some(
      (perm: { name: string; codename?: string; content_type: number }) => {
        const matchesName = perm.name === requiredPermission;
        const matchesCodename = perm.codename === requiredCodename || perm.codename === `can_${requiredCodename}`;
        this.logger.debug(`Checking permission: ${perm.name} (codename: ${perm.codename}, content_type: ${perm.content_type}) - MatchesName: ${matchesName}, MatchesCodename: ${matchesCodename}`);
        return matchesName || matchesCodename;
      }
    );

    // If requiredPermission is 'Can view employees', also allow if user has 'Can change Assigned To'
    if (!hasPermission && requiredPermission === 'Can view employees') {
      hasPermission = user.permissions.some(
        (perm: { name: string; codename?: string; content_type: number }) => {
          const matchesName = perm.name === 'Can change Assigned To';
          const matchesCodename = perm.codename === 'can_change_assigned_to';
          return matchesName || matchesCodename;
        }
      );
    }

    if (!hasPermission) {
      this.logger.error(`User lacks required permission: ${requiredPermission}`);
      throw new ForbiddenException(`User lacks required permission: ${requiredPermission}`);
    }

    // For user data access, add additional role-based restrictions
    if (requiredPermission === 'Can view employees') {
      // Store the requesting user in the request for use in the service layer
      request['requestingUser'] = user;
    }

    return true;
  }
}