"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrdersService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const order_entity_1 = require("./entities/order.entity");
const order_item_entity_1 = require("./entities/order-item.entity");
const customer_entity_1 = require("../customers/entities/customer.entity");
const product_entity_1 = require("../products/entities/product.entity");
const user_entity_1 = require("../users/entities/user.entity");
let OrdersService = class OrdersService {
    ordersRepository;
    orderItemsRepository;
    customersRepository;
    productsRepository;
    usersRepository;
    dataSource;
    constructor(ordersRepository, orderItemsRepository, customersRepository, productsRepository, usersRepository, dataSource) {
        this.ordersRepository = ordersRepository;
        this.orderItemsRepository = orderItemsRepository;
        this.customersRepository = customersRepository;
        this.productsRepository = productsRepository;
        this.usersRepository = usersRepository;
        this.dataSource = dataSource;
    }
    async generateOrderNumber() {
        const today = new Date();
        const dateStr = today.toISOString().split('T')[0].replace(/-/g, '');
        const lastOrder = await this.ordersRepository
            .createQueryBuilder('order')
            .where('order.orderNumber LIKE :pattern', { pattern: `ORD-${dateStr}-%` })
            .orderBy('order.createdAt', 'DESC')
            .getOne();
        let sequence = 1;
        if (lastOrder && lastOrder.orderNumber) {
            const parts = lastOrder.orderNumber.split('-');
            if (parts.length >= 3) {
                const lastSequence = parseInt(parts[2] || '0', 10);
                sequence = lastSequence + 1;
            }
        }
        const orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
        const exists = await this.ordersRepository.findOne({
            where: { orderNumber },
        });
        if (exists) {
            sequence++;
            return `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
        }
        return orderNumber;
    }
    async cleanupInvalidOrders() {
        try {
            await this.fixEmptyOrderNumbers();
            const allOrders = await this.ordersRepository.find();
            const allCustomerIds = (await this.customersRepository.find()).map(c => c.id);
            const invalidOrders = allOrders.filter(order => order.customerId !== null && order.customerId !== undefined && !allCustomerIds.includes(order.customerId));
            if (invalidOrders.length > 0) {
                console.log(`Found ${invalidOrders.length} orders with invalid customerIds. Removing...`);
                for (const order of invalidOrders) {
                    await this.orderItemsRepository.delete({ orderId: order.id });
                }
                await this.ordersRepository.remove(invalidOrders);
                console.log(`Removed ${invalidOrders.length} orders with invalid customerIds.`);
            }
        }
        catch (error) {
            console.warn('Failed to cleanup invalid orders:', error.message);
        }
    }
    async fixEmptyOrderNumbers() {
        try {
            const ordersWithoutNumber = await this.ordersRepository
                .createQueryBuilder('order')
                .where('order.orderNumber IS NULL OR order.orderNumber = :empty', { empty: '' })
                .getMany();
            if (ordersWithoutNumber.length > 0) {
                console.log(`Found ${ordersWithoutNumber.length} orders without order numbers. Fixing...`);
                for (const order of ordersWithoutNumber) {
                    const orderDate = order.createdAt || new Date();
                    const dateStr = orderDate.toISOString().split('T')[0].replace(/-/g, '');
                    const allOrders = await this.ordersRepository
                        .createQueryBuilder('o')
                        .where('o.orderNumber IS NOT NULL AND o.orderNumber != :empty', { empty: '' })
                        .andWhere('o.orderNumber LIKE :pattern', { pattern: `ORD-${dateStr}-%` })
                        .getMany();
                    let maxSequence = 0;
                    allOrders.forEach(o => {
                        if (o.orderNumber) {
                            const parts = o.orderNumber.split('-');
                            if (parts.length >= 3) {
                                const seq = parseInt(parts[2] || '0', 10);
                                if (seq > maxSequence)
                                    maxSequence = seq;
                            }
                        }
                    });
                    let sequence = maxSequence + 1;
                    let orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
                    let exists = await this.ordersRepository.findOne({
                        where: { orderNumber },
                    });
                    while (exists) {
                        sequence++;
                        orderNumber = `ORD-${dateStr}-${sequence.toString().padStart(4, '0')}`;
                        exists = await this.ordersRepository.findOne({
                            where: { orderNumber },
                        });
                    }
                    order.orderNumber = orderNumber;
                    await this.ordersRepository.save(order);
                }
                console.log(`Fixed ${ordersWithoutNumber.length} orders with missing order numbers.`);
            }
        }
        catch (error) {
            console.warn('Failed to fix empty order numbers:', error.message);
        }
    }
    async create(createOrderDto, user) {
        const customer = await this.customersRepository.findOne({
            where: { id: createOrderDto.customerId },
        });
        if (!customer) {
            throw new common_1.NotFoundException(`Customer with ID ${createOrderDto.customerId} not found`);
        }
        const productIds = createOrderDto.orderItems.map((item) => item.productId);
        const products = await this.productsRepository.find({
            where: productIds.map((id) => ({ id })),
        });
        if (products.length !== productIds.length) {
            const foundIds = products.map((p) => p.id);
            const missingIds = productIds.filter((id) => !foundIds.includes(id));
            throw new common_1.NotFoundException(`Products with IDs ${missingIds.join(', ')} not found`);
        }
        const orderNumber = await this.generateOrderNumber();
        if (!orderNumber) {
            throw new common_1.BadRequestException('Failed to generate order number');
        }
        if (!createOrderDto.customerId) {
            throw new common_1.BadRequestException('Customer ID is required');
        }
        const order = this.ordersRepository.create({
            orderNumber,
            customerId: createOrderDto.customerId,
            createdById: user.id,
        });
        const savedOrder = await this.ordersRepository.save(order);
        const orderItems = createOrderDto.orderItems.map((item) => this.orderItemsRepository.create({
            orderId: savedOrder.id,
            productId: item.productId,
            count: item.count,
        }));
        await this.orderItemsRepository.save(orderItems);
        return this.findOne(savedOrder.id, user);
    }
    async findAll(listOrdersDto, user) {
        const { page = 1, limit = 10, customerId, orderNumber, createdById, startDate, endDate, } = listOrdersDto;
        const skip = (page - 1) * limit;
        const queryBuilder = this.ordersRepository
            .createQueryBuilder('order')
            .leftJoinAndSelect('order.customer', 'customer')
            .leftJoinAndSelect('order.createdBy', 'createdBy')
            .leftJoinAndSelect('order.orderItems', 'orderItems')
            .leftJoinAndSelect('orderItems.product', 'product');
        if (customerId) {
            queryBuilder.where('order.customerId = :customerId', { customerId });
        }
        if (orderNumber) {
            if (customerId) {
                queryBuilder.andWhere('order.orderNumber LIKE :orderNumber', {
                    orderNumber: `%${orderNumber}%`,
                });
            }
            else {
                queryBuilder.where('order.orderNumber LIKE :orderNumber', {
                    orderNumber: `%${orderNumber}%`,
                });
            }
        }
        if (createdById) {
            const whereClause = customerId || orderNumber ? 'andWhere' : 'where';
            queryBuilder[whereClause]('order.createdById = :createdById', {
                createdById,
            });
        }
        if (startDate && endDate) {
            const whereClause = customerId || orderNumber || createdById ? 'andWhere' : 'where';
            queryBuilder[whereClause]('order.createdAt BETWEEN :startDate AND :endDate', {
                startDate,
                endDate,
            });
        }
        else if (startDate) {
            const whereClause = customerId || orderNumber || createdById ? 'andWhere' : 'where';
            queryBuilder[whereClause]('order.createdAt >= :startDate', { startDate });
        }
        else if (endDate) {
            const whereClause = customerId || orderNumber || createdById ? 'andWhere' : 'where';
            queryBuilder[whereClause]('order.createdAt <= :endDate', { endDate });
        }
        queryBuilder
            .andWhere('order.isActive = :isActive', { isActive: true })
            .skip(skip)
            .take(limit)
            .orderBy('order.createdAt', 'DESC');
        const [data, total] = await queryBuilder.getManyAndCount();
        return { data, total };
    }
    async findOne(id, user) {
        const order = await this.ordersRepository.findOne({
            where: { id },
            relations: ['customer', 'createdBy', 'orderItems', 'orderItems.product'],
        });
        if (!order) {
            throw new common_1.NotFoundException(`Order with ID ${id} not found`);
        }
        return order;
    }
    async update(id, updateOrderDto, user) {
        return await this.dataSource.transaction(async (manager) => {
            const orderRepository = manager.getRepository(order_entity_1.Order);
            const orderItemsRepository = manager.getRepository(order_item_entity_1.OrderItem);
            const customersRepository = manager.getRepository(customer_entity_1.Customer);
            const productsRepository = manager.getRepository(product_entity_1.Product);
            const order = await orderRepository.findOne({
                where: { id },
                relations: ['customer', 'createdBy'],
            });
            if (!order) {
                throw new common_1.NotFoundException(`Order with ID ${id} not found`);
            }
            if (updateOrderDto.customerId !== undefined) {
                const customer = await customersRepository.findOne({
                    where: { id: updateOrderDto.customerId },
                });
                if (!customer) {
                    throw new common_1.NotFoundException(`Customer with ID ${updateOrderDto.customerId} not found`);
                }
                order.customerId = updateOrderDto.customerId;
            }
            if (updateOrderDto.orderItems !== undefined) {
                const productIds = updateOrderDto.orderItems.map((item) => item.productId);
                const products = await productsRepository.find({
                    where: productIds.map((id) => ({ id })),
                });
                if (products.length !== productIds.length) {
                    const foundIds = products.map((p) => p.id);
                    const missingIds = productIds.filter((id) => !foundIds.includes(id));
                    throw new common_1.NotFoundException(`Products with IDs ${missingIds.join(', ')} not found`);
                }
                await orderItemsRepository.delete({ orderId: id });
                const orderItems = updateOrderDto.orderItems.map((item) => orderItemsRepository.create({
                    orderId: id,
                    productId: item.productId,
                    count: item.count,
                }));
                await orderItemsRepository.save(orderItems);
            }
            if (updateOrderDto.isActive !== undefined) {
                order.isActive = updateOrderDto.isActive;
            }
            await orderRepository.save(order);
            return this.findOne(id, user);
        });
    }
    async remove(id, user) {
        const order = await this.findOne(id, user);
        order.isActive = false;
        await this.ordersRepository.save(order);
    }
};
exports.OrdersService = OrdersService;
exports.OrdersService = OrdersService = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(order_entity_1.Order)),
    __param(1, (0, typeorm_1.InjectRepository)(order_item_entity_1.OrderItem)),
    __param(2, (0, typeorm_1.InjectRepository)(customer_entity_1.Customer)),
    __param(3, (0, typeorm_1.InjectRepository)(product_entity_1.Product)),
    __param(4, (0, typeorm_1.InjectRepository)(user_entity_1.User)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.DataSource])
], OrdersService);
//# sourceMappingURL=orders.service.js.map