import { Test, TestingModule } from '@nestjs/testing';
import { UsersController } from './users.controller';
import { UsersService } from './users.service';
import { getRepositoryToken } from '@nestjs/typeorm';
import { User } from './entities/user.entity';
import { Repository } from 'typeorm';
import { CreateUserDto } from './dto/create-user.dto';
import { UpdateUserDto } from './dto/update-user.dto';
import { NotFoundException } from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';

describe('UsersController', () => {
  let controller: UsersController;
  let service: UsersService;

  const mockUsersService = {
    create: jest.fn(),
    findAll: jest.fn(),
    findOne: jest.fn(),
    update: jest.fn(),
    remove: jest.fn(),
    deactivate: jest.fn(),
    reactivate: jest.fn(),
    findDeactivatedUsers: jest.fn(),
    bulkCreate: jest.fn(),
  };

  const mockJwtService = {
    sign: jest.fn(),
    verify: jest.fn(),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      controllers: [UsersController],
      providers: [
        {
          provide: UsersService,
          useValue: mockUsersService,
        },
        {
          provide: JwtService,
          useValue: mockJwtService,
        },
      ],
    }).compile();

    controller = module.get<UsersController>(UsersController);
    service = module.get<UsersService>(UsersService);
  });

  it('should be defined', () => {
    expect(controller).toBeDefined();
  });

  describe('create', () => {
    it('should create a new user', async () => {
      const createUserDto: CreateUserDto = {
        user: { email: 'test@example.com', name: 'John Doe' },
        role_id: '6',
        permissions: ['89', '90'],
        reports_to_id: '2',
      };
      const req = { user: { id: 1, role: 'Admin' } };
      const expectedUser = {
        id: 1,
        ...createUserDto,
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      };
      mockUsersService.create.mockResolvedValue(expectedUser);
      const result = await controller.create(createUserDto, req as any);
      expect(result).toEqual(expectedUser);
      expect(service.create).toHaveBeenCalledWith(createUserDto, req.user.id);
    });
  });

  describe('findAll', () => {
    it('should return an array of users', async () => {
      const expectedUsers = [
        {
          id: 1,
          user: { email: 'test@example.com', name: 'John Doe' },
          role_id: '6',
          permissions: ['89', '90'],
          reports_to_id: '2',
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date(),
        },
      ];
      mockUsersService.findAll.mockResolvedValue(expectedUsers);
      const req = { user: { id: 1, role: 'Admin' }, requestingUser: { id: 1, role: 'Admin' } };
      const result = await controller.findAll('10', '0', '', '', req as any);
      expect(result).toEqual(expectedUsers);
      expect(service.findAll).toHaveBeenCalled();
    });
  });

  describe('findOne', () => {
    it('should return a single user', async () => {
      const expectedUser = {
        id: 1,
        user: { email: 'test@example.com', name: 'John Doe' },
        role_id: '6',
        permissions: ['89', '90'],
        reports_to_id: '2',
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      };
      mockUsersService.findOne.mockResolvedValue(expectedUser);
      const req = { user: { id: 1, role: 'Admin' }, requestingUser: { id: 1, role: 'Admin' } };
      const result = await controller.findOne('1', req as any);
      expect(result).toEqual(expectedUser);
      expect(service.findOne).toHaveBeenCalledWith(1, req.requestingUser);
    });
  });

  describe('update', () => {
    it('should update a user', async () => {
      const updateUserDto: UpdateUserDto = {
        user: { email: 'test@example.com', name: 'Updated Name' },
      };
      const req = { user: { id: 1, role: 'Admin' } };
      const expectedUser = {
        id: 1,
        user: { email: 'test@example.com', name: 'Updated Name' },
        role_id: '6',
        permissions: ['89', '90'],
        reports_to_id: '2',
        isActive: true,
        createdAt: new Date(),
        updatedAt: new Date(),
      };
      mockUsersService.update.mockResolvedValue(expectedUser);
      const result = await controller.update('1', updateUserDto, req as any);
      expect(result).toEqual(expectedUser);
      expect(service.update).toHaveBeenCalledWith(1, updateUserDto);
    });
  });

  describe('remove', () => {
    it('should remove a user', async () => {
      const expectedResult = {
        status: 'success',
        message: 'User has been successfully deleted',
      };
      const req = { user: { id: 1, role: 'Admin' } };
      mockUsersService.remove.mockResolvedValue(expectedResult);
      const result = await controller.remove('1', req as any);
      expect(result).toEqual(expectedResult);
      expect(service.remove).toHaveBeenCalledWith(1);
    });
  });

  describe('deactivate', () => {
    it('should deactivate a user', async () => {
      const expectedResult = { user: { id: 1, isActive: false }, message: 'User deactivated' };
      mockUsersService.deactivate = jest.fn().mockResolvedValue(expectedResult);
      const req = { user: { id: 1, role: 'Admin' } };
      const result = await controller.deactivate('1', req as any);
      expect(result).toEqual(expectedResult);
      expect(service.deactivate).toHaveBeenCalledWith(1);
    });
  });

  describe('reactivate', () => {
    it('should reactivate a user', async () => {
      const expectedResult = { user: { id: 1, isActive: true }, message: 'User reactivated' };
      mockUsersService.reactivate = jest.fn().mockResolvedValue(expectedResult);
      const req = { user: { id: 1, role: 'Admin' } };
      const result = await controller.reactivate('1', req as any);
      expect(result).toEqual(expectedResult);
      expect(service.reactivate).toHaveBeenCalledWith(1);
    });
  });

  describe('findDeactivatedUsers', () => {
    it('should return deactivated users', async () => {
      const expectedUsers = [
        { id: 1, isActive: false },
        { id: 2, isActive: false },
      ];
      mockUsersService.findDeactivatedUsers = jest.fn().mockResolvedValue(expectedUsers);
      const req = { user: { id: 1, role: 'Admin' } };
      const result = await controller.findDeactivatedUsers(req as any);
      expect(result).toEqual(expectedUsers);
      expect(service.findDeactivatedUsers).toHaveBeenCalled();
    });
  });

  describe('bulkCreate', () => {
    it('should bulk create users from a valid file', async () => {
      const expectedResult = { message: 'Bulk create success', details: { success: 2, failed: 0, errors: [] } };
      mockUsersService.bulkCreate = jest.fn().mockResolvedValue(expectedResult);
      const file = { buffer: Buffer.from(''), originalname: 'users.xlsx' } as any;
      const req = { user: { id: 1, role: 'Admin' } };
      const result = await controller.bulkCreate(file, req as any);
      expect(result).toEqual(expectedResult);
      expect(service.bulkCreate).toHaveBeenCalledWith(file);
    });
    it('should throw BadRequestException for invalid file', async () => {
      mockUsersService.bulkCreate = jest.fn().mockRejectedValue(new Error('Invalid file'));
      const file = { buffer: Buffer.from(''), originalname: 'invalid.txt' } as any;
      const req = { user: { id: 1, role: 'Admin' } };
      await expect(controller.bulkCreate(file, req as any)).rejects.toThrow('Invalid file');
    });
  });
}); 