import { Test, TestingModule } from '@nestjs/testing';
import { UserActivityService } from './user-activity.service';
import { getRepositoryToken } from '@nestjs/typeorm';
import { UserActivity } from './entities/user-activity.entity';
import { Repository } from 'typeorm';
import { User } from '../users/entities/user.entity';

describe('UserActivityService', () => {
  let service: UserActivityService;
  let repository: Repository<UserActivity>;

  const mockRepository = {
    create: jest.fn(),
    save: jest.fn(),
    findAndCount: jest.fn(),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        UserActivityService,
        {
          provide: getRepositoryToken(UserActivity),
          useValue: mockRepository,
        },
      ],
    }).compile();

    service = module.get<UserActivityService>(UserActivityService);
    repository = module.get<Repository<UserActivity>>(getRepositoryToken(UserActivity));
  });

  it('should be defined', () => {
    expect(service).toBeDefined();
  });

  describe('logActivity', () => {
    it('should log user activity', async () => {
      const mockUser: User = {
        id: 1,
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        role: 'ADMIN',
        isActive: true,
        status: 1,
        createdAt: new Date(),
        updatedAt: new Date(),
        lastLoginAt: new Date(),
      } as User;

      const activityParams = {
        user: mockUser,
        action: 'CREATE',
        type: 'CUSTOMER',
        ipAddress: '127.0.0.1',
        changedObject: 'Customer',
        method: 'POST',
      };

      const mockActivity = {
        id: 1,
        username: 'test@example.com',
        action: 'CREATE',
        type: 'CUSTOMER',
        ipAddress: '127.0.0.1',
        changedObject: 'Customer',
        method: 'POST',
        time: new Date(),
        user: mockUser,
      };

      mockRepository.create.mockReturnValue(mockActivity);
      mockRepository.save.mockResolvedValue(mockActivity);

      await service.logActivity(activityParams);

      expect(repository.create).toHaveBeenCalledWith({
        username: 'test@example.com',
        action: 'CREATE',
        type: 'CUSTOMER',
        ipAddress: '127.0.0.1',
        changedObject: 'Customer',
        method: 'POST',
        time: expect.any(Date),
        user: mockUser,
      });
      expect(repository.save).toHaveBeenCalledWith(mockActivity);
    });
  });

  describe('getUserActivities', () => {
    it('should return user activities with pagination', async () => {
      const mockUser: User = {
        id: 1,
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        role: 'ADMIN',
        isActive: true,
        status: 1,
        createdAt: new Date(),
        updatedAt: new Date(),
        lastLoginAt: new Date(),
      } as User;

      const mockActivities = [
        {
          id: 1,
          user: mockUser,
          action: 'CREATE',
          method: 'POST',
          userAgent: 'Mozilla/5.0',
          ipAddress: '127.0.0.1',
          changes: 'Created new customer',
          changedObject: 'Customer',
          time: new Date(),
          type: 'CUSTOMER',
        },
      ];

      const expectedResponse = {
        previous: null,
        count: 1,
        limit: 10,
        offset: 0,
        status: 200,
        success: true,
        message: 'List Successfully Retrieved!',
        data: [
          {
            id: 1,
            user: {
              email: 'test@example.com',
              name: 'John Doe',
              last_login: mockUser.lastLoginAt?.toISOString(),
            },
            action: 'CREATE',
            method: 'POST',
            user_agent: 'Mozilla/5.0',
            ip_address: '127.0.0.1',
            changes: 'Created new customer',
            changed_object: 'Customer',
            time: mockActivities[0].time.toISOString(),
            type: 'CUSTOMER',
          },
        ],
      };

      mockRepository.findAndCount.mockResolvedValue([mockActivities, 1]);

      const result = await service.getUserActivities(1, 10, 0);

      expect(result).toEqual(expectedResponse);
      expect(repository.findAndCount).toHaveBeenCalledWith({
        where: { user: { id: 1 } },
        relations: ['user'],
        order: { time: 'DESC' },
        take: 10,
        skip: 0,
      });
    });

    it('should return previous page URL when offset > 0', async () => {
      const mockUser: User = {
        id: 1,
        email: 'test@example.com',
        firstName: 'John',
        lastName: 'Doe',
        role: 'ADMIN',
        isActive: true,
        status: 1,
        createdAt: new Date(),
        updatedAt: new Date(),
        lastLoginAt: new Date(),
      } as User;

      const mockActivities = [
        {
          id: 1,
          user: mockUser,
          action: 'CREATE',
          method: 'POST',
          userAgent: 'Mozilla/5.0',
          ipAddress: '127.0.0.1',
          changes: 'Created new customer',
          changedObject: 'Customer',
          time: new Date(),
          type: 'CUSTOMER',
        },
      ];

      mockRepository.findAndCount.mockResolvedValue([mockActivities, 1]);

      const result = await service.getUserActivities(1, 10, 20);

      expect(result.previous).toBe('/api/core/users_activity/?limit=10&offset=10&user__id=1');
      expect(result.count).toBe(1);
      expect(result.limit).toBe(10);
      expect(result.offset).toBe(20);
    });
  });
});
