import {
  Injectable,
  NotFoundException,
  UnauthorizedException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Login } from './entities/login.entity';
import { User } from '../users/entities/user.entity';
import { EmailService } from '../common/services/email.service';
import { JwtService } from '@nestjs/jwt';
import { Request } from 'express';
import { UserActivityService } from '../user-activity/user-activity.service';
import { PermissionsService } from '../permissions/permissions.service';
import { PermissionResponse } from '../permissions/permissions.model';

@Injectable()
export class LoginService {
  constructor(
    @InjectRepository(Login)
    private readonly loginRepository: Repository<Login>,

    @InjectRepository(User)
    private readonly userRepository: Repository<User>,

    private readonly emailService: EmailService,
    private readonly jwtService: JwtService,
    private readonly userActivityService: UserActivityService,
    private readonly permissionsService: PermissionsService,
  ) {}

  private getClientIp(request: Request): string {
    const forwarded = request.headers['x-forwarded-for'];
    if (typeof forwarded === 'string') return forwarded.split(',')[0].trim();

    const ip =
      request.socket?.remoteAddress ||
      request.connection?.remoteAddress ||
      request.ip ||
      'unknown';

    if (ip === '::1' || ip === '127.0.0.1') return '127.0.0.1';
    if (ip.startsWith('::ffff:')) return ip.replace('::ffff:', '');
    return ip;
  }

  async requestOTP(email: string, request: Request): Promise<{ message: string }> {
    const user = await this.userRepository.findOne({ where: { email } });

    if (!user) {
      await this.loginRepository.save({
        email,
        status: 'failed',
        failureReason: 'Invalid Email',
        ipAddress: this.getClientIp(request),
      });
      throw new NotFoundException('Invalid Email');
    }

    const otp = Math.floor(10000 + Math.random() * 90000).toString();
    const otpExpiresAt = new Date();
    otpExpiresAt.setMinutes(otpExpiresAt.getMinutes() + 5);

    const loginAttempt = await this.loginRepository.save({
      email,
      otp,
      otpExpiresAt,
      isVerified: false,
      status: 'pending',
      ipAddress: this.getClientIp(request),
      user,
    });

    try {
      await this.emailService.sendOTP(email, otp);
    } catch (error) {
      await this.loginRepository.update(loginAttempt.id, {
        status: 'failed',
        failureReason: 'Failed to send OTP email',
      });
      console.error('Failed to send OTP email:', error);
      throw new Error('Failed to send OTP email');
    }

    return { message: 'OTP sent successfully' };
  }

  async verifyOTP(
    email: string,
    otp: string,
    request: Request,
  ): Promise<{ user: any; accessToken: string; refreshToken: string }> {
    const loginAttempt = await this.loginRepository.findOne({
      where: { email, otp },
      order: { createdAt: 'DESC' },
    });

    if (!loginAttempt) {
      await this.loginRepository.save({
        email,
        status: 'failed',
        failureReason: 'Invalid OTP',
        ipAddress: this.getClientIp(request),
      });
      throw new UnauthorizedException('Invalid OTP');
    }

    if (loginAttempt.otpExpiresAt < new Date()) {
      await this.loginRepository.update(loginAttempt.id, {
        status: 'failed',
        failureReason: 'OTP expired',
      });
      throw new UnauthorizedException('OTP has expired');
    }

    if (loginAttempt.isVerified) {
      await this.loginRepository.update(loginAttempt.id, {
        status: 'failed',
        failureReason: 'OTP already used',
      });
      throw new UnauthorizedException('OTP already used');
    }

    await this.loginRepository.update(loginAttempt.id, {
      isVerified: true,
      status: 'success',
      verifiedAt: new Date(),
    });

    const user = await this.userRepository.findOne({ where: { email } });
    if (!user) throw new NotFoundException('User not found');

    // Get user-specific permissions
    let userPermissions: PermissionResponse[] = [];
    if (user.permissions) {
      try {
        // If permissions is a string, parse it
        if (typeof user.permissions === 'string') {
          userPermissions = JSON.parse(user.permissions);
        } else {
          userPermissions = user.permissions;
        }
      } catch (error) {
        console.error(`Error parsing permissions for user ${user.id}:`, error);
        userPermissions = [];
      }
    }

    await this.userActivityService.logActivity({
      user,
      action: 'Login Verified',
      type: 'login',
      ipAddress: this.getClientIp(request),
      changedObject: `User ${user.email} successfully logged in.`,
      method: request.method,
    });

    const tokenPayload = {
      sub: user.id,
      email: user.email,
      role: user.role,
      permissions: userPermissions
    };

    const accessToken = this.generateAccessToken(tokenPayload);
    const refreshToken = this.generateRefreshToken(tokenPayload);

    return {
      user: {
        id: user.id,
        email: user.email,
        firstName: user.firstName,
        lastName: user.lastName,
        role: user.role,
        permissions: userPermissions
      },
      accessToken,
      refreshToken
    };
  }

  private getRoleIdFromName(roleName: string): string {
    const roleMap: { [key: string]: string } = {
      'Admin': '1',
      'Retention': '2',
      'Reactivation': '3',
      'Owner': '4',
      'Escalator': '5',
      'Agent': '6',
      'Team Lead': '7'
    };
    return roleMap[roleName] || '1';
  }

  generateAccessToken(payload: any): string {
    return this.jwtService.sign(payload, {
      expiresIn: '10d',
    });
  }

  generateRefreshToken(payload: any): string {
    return this.jwtService.sign(payload, {
      expiresIn: '7d'
    });
  }
}
