import { IsString, IsOptional, IsEmail, IsArray, IsBoolean, IsInt, ValidateNested, Validate } from 'class-validator';
import { Transform, Type } from 'class-transformer';

class AddressDto {
  @IsInt()
  @IsOptional()
  id?: number;

  @IsString()
  @IsOptional()
  address1?: string;

  @IsString()
  @IsOptional()
  address2?: string;

  @IsString()
  @IsOptional()
  town_city?: string;

  @IsString()
  @IsOptional()
  county?: string;

  @IsString()
  @IsOptional()
  country?: string;

  @IsString()
  @IsOptional()
  post_code?: string;

  @IsBoolean()
  @IsOptional()
  primary?: boolean;

  @IsString()
  @IsOptional()
  formatted_address?: string;

  @IsBoolean()
  @IsOptional()
  is_active?: boolean;

  @Validate((obj: AddressDto) => {
    return (
      !!obj.id ||
      !!obj.address1 ||
      !!obj.address2 ||
      !!obj.town_city ||
      !!obj.county ||
      !!obj.country ||
      !!obj.post_code ||
      !!obj.formatted_address ||
      obj.primary !== undefined ||
      obj.is_active !== undefined
    );
  }, { message: 'At least one address field must be provided' })
  hasAtLeastOneField: boolean;
}

export class UpdateCustomerDto {
  @IsString()
  @IsOptional()
  @Transform(({ value }) => value?.trim())
  first_name?: string;

  @IsString()
  @IsOptional()
  @Transform(({ value }) => value?.trim())
  last_name?: string;

  @IsEmail({}, { message: 'Invalid email format' })
  @IsOptional()
  @Transform(({ value }) => value?.trim().toLowerCase())
  email?: string;

  @IsString()
  @IsOptional()
  @Transform(({ value }) => value?.trim())
  mobile_phone?: string;

  @IsString()
  @IsOptional()
  @Transform(({ value }) => value?.trim())
  home_phone?: string;

  @IsArray()
  @IsOptional()
  @ValidateNested({ each: true })
  @Type(() => AddressDto)
  address?: AddressDto[];

  @IsString()
  @IsOptional()
  title?: string;

  @IsString()
  @IsOptional()
  iso?: string;

  @IsBoolean()
  @IsOptional()
  DNC?: boolean;

  @IsBoolean()
  @IsOptional()
  is_land_line_default?: boolean;

  @IsBoolean()
  @IsOptional()
  is_active?: boolean;

  @IsString()
  @IsOptional()
  customer_id?: string;

  @IsString()
  @IsOptional()
  gen_customer_id?: string;

  @IsInt()
  @IsOptional()
  @Transform(({ value }) => (value ? parseInt(value, 10) : undefined))
  assigned_to_id?: number;
}
