import { Test, TestingModule } from '@nestjs/testing';
import { CustomersService } from './customers.service';
import { getRepositoryToken } from '@nestjs/typeorm';
import { Customer } from './entities/customer.entity';
import { Repository, Like } from 'typeorm';
import { CreateCustomerDto } from './dto/create-customer.dto';
import { ListCustomersDto } from './dto/list-customers.dto';
import { ConflictException } from '@nestjs/common';
import { Address } from './entities/address.entity';
import { CustomerNote } from './entities/customer-note.entity';
import { User } from '../users/entities/user.entity';

describe('CustomersService', () => {
  let service: CustomersService;
  let repo: Repository<Customer>;

const mockCustomer = {
  id: 1,
  firstName: 'Ramesh',
  lastName: 'Takoba',
  email: 'rohitznaik78924@gmail.com',
  mobileNumber: '9945566683',
  landline: '07892269418',
  address: [],
  title: 'Ms',
  iso: 'GB-ENG',
  dnc: false,
  isLandLineDefault: true,
  createdAt: new Date(),
};

  const mockTransformedCustomer = {
    id: 1,
    first_name: 'Ramesh',
    last_name: 'Takoba',
    email: 'rohitznaik78924@gmail.com',
    mobile_phone: '9945566683',
    home_phone: '07892269418',
    address: [],
    title: 'Ms',
    iso: 'GB-ENG',
    DNC: false,
    is_land_line_default: true,
    created_at: mockCustomer.createdAt,
  };

  const mockQueryBuilder = {
    leftJoinAndSelect: jest.fn().mockReturnThis(),
    skip: jest.fn().mockReturnThis(),
    take: jest.fn().mockReturnThis(),
    orderBy: jest.fn().mockReturnThis(),
    andWhere: jest.fn().mockReturnThis(),
    getManyAndCount: jest.fn().mockResolvedValue([[mockCustomer], 1]),
  };

  const mockRepository = {
    findOne: jest.fn(),
    create: jest.fn().mockReturnValue(mockCustomer),
    save: jest.fn().mockResolvedValue(mockCustomer),
    findAndCount: jest.fn().mockResolvedValue([[mockCustomer], 1]),
    createQueryBuilder: jest.fn(() => mockQueryBuilder),
    merge: jest.fn(),
  };

  const mockUserRepository = { 
    findOne: jest.fn(),
    find: jest.fn().mockResolvedValue([]),
  };
  const mockAddressRepository = {};
  const mockCustomerNoteRepository = {};

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        CustomersService,
        {
          provide: getRepositoryToken(Customer),
          useValue: mockRepository,
        },
        {
          provide: getRepositoryToken(Address),
          useValue: mockAddressRepository,
        },
        {
          provide: getRepositoryToken(CustomerNote),
          useValue: mockCustomerNoteRepository,
        },
        {
          provide: getRepositoryToken(User),
          useValue: mockUserRepository,
        },
      ],
    }).compile();

    service = module.get<CustomersService>(CustomersService);
    repo = module.get<Repository<Customer>>(getRepositoryToken(Customer));
  });

  it('should be defined', () => {
    expect(service).toBeDefined();
  });

  describe('create', () => {
    it('should create a new customer if email does not exist', async () => {
      mockRepository.findOne.mockResolvedValue(null);
      const dto: CreateCustomerDto = {
        first_name: 'Ramesh',
        last_name: 'Takoba',
        email: 'rohitznaik78924@gmail.com',
        mobile_phone: '9945566683',
        home_phone: '07892269418',
        address: [],
        title: 'Ms',
        iso: 'GB-ENG',
        DNC: false,
        is_land_line_default: true,
        assigned_to_id: 1,
      };
      const user = { id: 1, role: 'admin' };
      mockUserRepository.findOne.mockResolvedValue(user);
      const result = await service.create(dto, user);
      expect(mockRepository.findOne).toHaveBeenCalledWith({ where: { email: dto.email } });
      expect(mockRepository.create).toHaveBeenCalledWith({
        firstName: dto.first_name,
        lastName: dto.last_name,
        email: dto.email,
        mobileNumber: dto.mobile_phone,
        landline: dto.home_phone,
        title: dto.title,
        iso: dto.iso,
        dnc: dto.DNC,
        isLandLineDefault: dto.is_land_line_default,
        assigned_to_id: dto.assigned_to_id,
        createdById: user.id,
      });
      expect(mockRepository.save).toHaveBeenCalledWith(mockCustomer);
      expect(result).toEqual(mockCustomer);
    });

    it('should throw ConflictException if email already exists', async () => {
      mockRepository.findOne.mockResolvedValue(mockCustomer);
      const dto: CreateCustomerDto = {
        first_name: 'Ramesh',
        last_name: 'Takoba',
        email: 'rohitznaik78924@gmail.com',
        mobile_phone: '9945566683',
        home_phone: '07892269418',
        address: [],
        title: 'Ms',
        iso: 'GB-ENG',
        DNC: false,
        is_land_line_default: true,
        assigned_to_id: 1,
      };
      const user = { id: 1, role: 'admin' };
      await expect(service.create(dto, user)).rejects.toThrow(ConflictException);
      expect(mockRepository.findOne).toHaveBeenCalledWith({ where: { email: dto.email } });
    });
  });

  describe('findAll', () => {
    it('should return a list of customers with pagination', async () => {
      const listCustomersDto: ListCustomersDto = {
        offset: 0,
        limit: 10,
        search: '',
        scope: '',
      };
      const user = { id: 1, role: 'admin' };
      const result = await service.findAll(listCustomersDto, user);
      expect(mockQueryBuilder.getManyAndCount).toHaveBeenCalled();
      expect(result.data[0]).toEqual(expect.objectContaining({
        id: 1,
        first_name: 'Ramesh',
        last_name: 'Takoba',
        email: 'rohitznaik78924@gmail.com',
        mobile_phone: '9945566683',
        home_phone: '07892269418',
        title: 'Ms',
        iso: 'GB-ENG',
      }));
      expect(result.total).toBe(1);
    });

    it('should return a list of customers with search', async () => {
      const listCustomersDto: ListCustomersDto = {
        offset: 0,
        limit: 10,
        search: 'Ramesh',
        scope: '',
      };
      const user = { id: 1, role: 'admin' };
      const result = await service.findAll(listCustomersDto, user);
      expect(mockQueryBuilder.getManyAndCount).toHaveBeenCalled();
      expect(result.data[0]).toEqual(expect.objectContaining({
        id: 1,
        first_name: 'Ramesh',
        last_name: 'Takoba',
        email: 'rohitznaik78924@gmail.com',
        mobile_phone: '9945566683',
        home_phone: '07892269418',
        title: 'Ms',
        iso: 'GB-ENG',
      }));
      expect(result.total).toBe(1);
    });
  });

  describe('findOne', () => {
    it('should return a customer if found and authorized', async () => {
      const user = { id: 1, role: 'admin' };
      const customer = { id: 1, assigned_to_id: 1, createdById: 1, addresses: [], assignedTo: null, dnc: false, isLandLineDefault: true, iso: '', title: '', firstName: 'Test', lastName: 'User', email: 'test@example.com', mobileNumber: '', landline: '', createdAt: new Date(), updatedAt: new Date(), isActive: true, customerId: '', genCustomerId: '' };
      mockRepository.findOne.mockResolvedValue(customer);
      const customerNotesRepository = { find: jest.fn().mockResolvedValue([]) };
      const usersRepository = { findOne: jest.fn().mockResolvedValue(user) };
      Object.defineProperty(service, 'customerNotesRepository', { value: customerNotesRepository });
      Object.defineProperty(service, 'usersRepository', { value: usersRepository });
      const result = await service.findOne(1, user);
      expect(result.id).toBe(1);
    });
    it('should throw NotFoundException if customer not found', async () => {
      const user = { id: 1, role: 'admin' };
      mockRepository.findOne.mockResolvedValue(null);
      await expect(service.findOne(1, user)).rejects.toThrow('Customer with ID 1 not found');
    });
    it('should throw UnauthorizedException if user not authenticated', async () => {
      await expect(service.findOne(1, null)).rejects.toThrow('User must be authenticated');
    });
  });

  describe('updateById', () => {
    it('should update and return a customer if authorized', async () => {
      const user = { id: 1, role: 'admin' };
      const customer = { id: 1, assigned_to_id: 1, createdById: 1, addresses: [], email: 'old@example.com', genCustomerId: 'GEN1' };
      mockRepository.findOne.mockImplementationOnce(() => Promise.resolve(customer));
      mockRepository.findOne.mockImplementationOnce(() => Promise.resolve(null));
      mockRepository.findOne.mockImplementationOnce(() => Promise.resolve(null));
      mockRepository.save.mockResolvedValue({ ...customer, email: 'new@example.com' });
      mockRepository.merge.mockReturnValue({ ...customer, email: 'new@example.com', addresses: [] });
      const usersRepository = { findOne: jest.fn().mockResolvedValue(user) };
      Object.defineProperty(service, 'usersRepository', { value: usersRepository });
      const dto = {
        first_name: '',
        last_name: '',
        email: 'new@example.com',
        mobile_phone: '',
        home_phone: '',
        address: [],
        title: '',
        iso: '',
        DNC: false,
        is_land_line_default: false,
        is_active: true,
        customer_id: '',
        gen_customer_id: 'GEN1',
        assigned_to_id: 1,
      };
      const result = await service.updateById(1, dto, user);
      expect(result.email).toBe('new@example.com');
    });
    it('should throw NotFoundException if customer not found', async () => {
      const user = { id: 1, role: 'admin' };
      mockRepository.findOne.mockResolvedValue(null);
      await expect(service.updateById(1, {
        first_name: '', last_name: '', email: '', mobile_phone: '', home_phone: '', address: [], title: '', iso: '', DNC: false, is_land_line_default: false, is_active: true, customer_id: '', gen_customer_id: '', assigned_to_id: 1
      }, user)).rejects.toThrow('Customer with id 1 not found');
    });
    it('should throw UnauthorizedException if user not authenticated', async () => {
      await expect(service.updateById(1, {
        first_name: '', last_name: '', email: '', mobile_phone: '', home_phone: '', address: [], title: '', iso: '', DNC: false, is_land_line_default: false, is_active: true, customer_id: '', gen_customer_id: '', assigned_to_id: 1
      }, null)).rejects.toThrow('User must be authenticated');
    });
  });

  describe('bulkAssign', () => {
    it('should assign customers to a user', async () => {
      const user = { id: 1, role: 'admin' };
      const assignee = { id: 2, firstName: 'John', lastName: 'Doe' };
      const customers = [{ id: 1 }, { id: 2 }];
      const usersRepository = { findOne: jest.fn().mockResolvedValue(assignee) };
      const customersRepository = { findBy: jest.fn().mockResolvedValue(customers), save: jest.fn().mockResolvedValue(customers) };
      Object.defineProperty(service, 'usersRepository', { value: usersRepository });
      Object.defineProperty(service, 'customersRepository', { value: customersRepository });
      const dto = { customer_ids: [1, 2], assigned_to_id: 2 };
      const result = await service.bulkAssign(dto, user);
      expect(result).toContain('John_Doe');
    });
    it('should throw NotFoundException if assignee not found', async () => {
      const user = { id: 1, role: 'admin' };
      const usersRepository = { findOne: jest.fn().mockResolvedValue(null) };
      Object.defineProperty(service, 'usersRepository', { value: usersRepository });
      const dto = { customer_ids: [1, 2], assigned_to_id: 2 };
      await expect(service.bulkAssign(dto, user)).rejects.toThrow('User with ID 2 not found');
    });
    it('should throw NotFoundException if no customers found', async () => {
      const user = { id: 1, role: 'admin' };
      const assignee = { id: 2, firstName: 'John', lastName: 'Doe' };
      const usersRepository = { findOne: jest.fn().mockResolvedValue(assignee) };
      const customersRepository = { findBy: jest.fn().mockResolvedValue([]) };
      Object.defineProperty(service, 'usersRepository', { value: usersRepository });
      Object.defineProperty(service, 'customersRepository', { value: customersRepository });
      const dto = { customer_ids: [1, 2], assigned_to_id: 2 };
      await expect(service.bulkAssign(dto, user)).rejects.toThrow('No customers found with the provided IDs');
    });
  });
});