import { Injectable, Logger } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, LessThan } from 'typeorm';
import { JwtTokenActivity } from '../entities/jwt-token-activity.entity';
import { Request } from 'express';

@Injectable()
export class JwtTokenActivityService {
  private readonly logger = new Logger(JwtTokenActivityService.name);

  constructor(
    @InjectRepository(JwtTokenActivity)
    private jwtTokenActivityRepository: Repository<JwtTokenActivity>,
  ) {}

  private logTokenExpiration(token: string, userId: number, duration: number) {
    const expirationMessage = `
╔════════════════════════════════════════════════════════════════════════════╗
║                           TOKEN EXPIRATION ALERT                           ║
╠════════════════════════════════════════════════════════════════════════════╣
║ Token: ${token.substring(0, 20)}...                                        ║
║ User ID: ${userId}                                                         ║
║ Duration Used: ${duration} minutes                                         ║
║ Expired At: ${new Date().toISOString()}                                   ║
╚════════════════════════════════════════════════════════════════════════════╝
`;
    console.log(expirationMessage);
    this.logger.warn(`Token expired for user ${userId} after ${duration} minutes of use`);
  }

  async logTokenActivity(token: string, userId: number, req: Request): Promise<void> {
    try {
      const expiresAt = new Date();
      expiresAt.setMinutes(expiresAt.getMinutes() + 15); // 15 minutes expiration

      const tokenActivity = this.jwtTokenActivityRepository.create({
        token,
        user_id: userId,
        expiresAt,
        ipAddress: req.ip,
        userAgent: req.headers['user-agent'],
        lastUsedAt: new Date(),
      });

      await this.jwtTokenActivityRepository.save(tokenActivity);
      
      const creationMessage = `
╔════════════════════════════════════════════════════════════════════════════╗
║                           NEW TOKEN CREATED                                ║
╠════════════════════════════════════════════════════════════════════════════╣
║ Token: ${token.substring(0, 20)}...                                        ║
║ User ID: ${userId}                                                         ║
║ Created At: ${new Date().toISOString()}                                   ║
║ Expires At: ${expiresAt.toISOString()}                                    ║
╚════════════════════════════════════════════════════════════════════════════╝
`;
      console.log(creationMessage);
      this.logger.debug(`Token activity logged for user ${userId}. Expires at: ${expiresAt.toISOString()}`);
    } catch (error) {
      this.logger.error(`Failed to log token activity: ${error.message}`);
    }
  }

  async updateTokenLastUsed(token: string): Promise<void> {
    try {
      const tokenActivity = await this.jwtTokenActivityRepository.findOne({
        where: { token, isValid: true }
      });

      if (!tokenActivity) {
        this.logger.warn(`Token not found or invalid: ${token}`);
        return;
      }

      const now = new Date();
      const timeUntilExpiry = tokenActivity.expiresAt.getTime() - now.getTime();
      const minutesUntilExpiry = Math.floor(timeUntilExpiry / (1000 * 60));

      if (timeUntilExpiry <= 0) {
        const duration = now.getTime() - tokenActivity.createdAt.getTime();
        const minutesUsed = Math.floor(duration / (1000 * 60));
        this.logTokenExpiration(token, tokenActivity.user_id, minutesUsed);
        await this.markTokenAsExpired(token);
        return;
      }

      const usageMessage = `
╔════════════════════════════════════════════════════════════════════════════╗
║                           TOKEN USAGE UPDATE                               ║
╠════════════════════════════════════════════════════════════════════════════╣
║ Token: ${token.substring(0, 20)}...                                        ║
║ User ID: ${tokenActivity.user_id}                                          ║
║ Minutes Until Expiry: ${minutesUntilExpiry}                                ║
║ Last Used At: ${now.toISOString()}                                        ║
╚════════════════════════════════════════════════════════════════════════════╝
`;
      console.log(usageMessage);

      await this.jwtTokenActivityRepository.update(
        { token, isValid: true },
        { lastUsedAt: now }
      );
    } catch (error) {
      this.logger.error(`Failed to update token last used: ${error.message}`);
    }
  }

  async invalidateToken(token: string): Promise<void> {
    try {
      const tokenActivity = await this.jwtTokenActivityRepository.findOne({
        where: { token }
      });

      if (!tokenActivity) {
        this.logger.warn(`Token not found for invalidation: ${token}`);
        return;
      }

      const duration = new Date().getTime() - tokenActivity.createdAt.getTime();
      const minutesUsed = Math.floor(duration / (1000 * 60));

      const invalidationMessage = `
╔════════════════════════════════════════════════════════════════════════════╗
║                           TOKEN INVALIDATED                                ║
╠════════════════════════════════════════════════════════════════════════════╣
║ Token: ${token.substring(0, 20)}...                                        ║
║ User ID: ${tokenActivity.user_id}                                          ║
║ Duration Used: ${minutesUsed} minutes                                      ║
║ Invalidated At: ${new Date().toISOString()}                               ║
╚════════════════════════════════════════════════════════════════════════════╝
`;
      console.log(invalidationMessage);

      await this.jwtTokenActivityRepository.update(
        { token },
        { isValid: false }
      );
    } catch (error) {
      this.logger.error(`Failed to invalidate token: ${error.message}`);
    }
  }

  async markTokenAsExpired(token: string): Promise<void> {
    try {
      const tokenActivity = await this.jwtTokenActivityRepository.findOne({
        where: { token }
      });

      if (!tokenActivity) {
        this.logger.warn(`Token not found for expiration: ${token}`);
        return;
      }

      const duration = new Date().getTime() - tokenActivity.createdAt.getTime();
      const minutesUsed = Math.floor(duration / (1000 * 60));

      this.logTokenExpiration(token, tokenActivity.user_id, minutesUsed);

      await this.jwtTokenActivityRepository.update(
        { token },
        { isExpired: true, isValid: false }
      );
    } catch (error) {
      this.logger.error(`Failed to mark token as expired: ${error.message}`);
    }
  }

  async cleanupExpiredTokens(): Promise<void> {
    try {
      const tenDaysAgo = new Date();
      tenDaysAgo.setDate(tenDaysAgo.getDate() - 10);

      const result = await this.jwtTokenActivityRepository.delete({
        createdAt: LessThan(tenDaysAgo)
      });

      this.logger.debug(`Cleaned up ${result.affected} expired tokens older than 10 days`);
    } catch (error) {
      this.logger.error(`Failed to cleanup expired tokens: ${error.message}`);
    }
  }

  async getTokenActivities(userId: number, limit: number = 10, offset: number = 0) {
    try {
      const [activities, total] = await this.jwtTokenActivityRepository.findAndCount({
        where: { user_id: userId },
        order: { createdAt: 'DESC' },
        take: limit,
        skip: offset,
      });

      // Add duration information to each activity
      const activitiesWithDuration = activities.map(activity => {
        const duration = activity.lastUsedAt 
          ? activity.lastUsedAt.getTime() - activity.createdAt.getTime()
          : 0;
        const minutesUsed = Math.floor(duration / (1000 * 60));
        
        return {
          ...activity,
          duration_minutes: minutesUsed,
          time_until_expiry: activity.expiresAt.getTime() - new Date().getTime(),
        };
      });

      this.logger.debug(`Retrieved ${activities.length} token activities for user ${userId}`);

      return {
        data: activitiesWithDuration,
        total,
        limit,
        offset,
      };
    } catch (error) {
      this.logger.error(`Failed to get token activities: ${error.message}`);
      throw error;
    }
  }
} 