import { Injectable, UnauthorizedException, Logger } from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import { JwtTokenActivityService } from './services/jwt-token-activity.service';
import { Request } from 'express';

@Injectable()
export class AuthService {
  private readonly logger = new Logger(AuthService.name);

  constructor(
    private jwtService: JwtService,
    private jwtTokenActivityService: JwtTokenActivityService,
  ) {}

  async validateToken(token: string, req: Request) {
    try {
      const payload = await this.jwtService.verifyAsync(token);
      
      // Check token activity
      const tokenActivity = await this.jwtTokenActivityService.updateTokenLastUsed(token);
      
      this.logger.debug(`Token validated for user ${payload.sub}. Token: ${token.substring(0, 10)}...`);
      
      return {
        isValid: true,
        user: {
          id: payload.sub,
          email: payload.email,
          role: payload.role,
          permissions: payload.permissions || []
        },
      };
    } catch (error) {
      this.logger.error(`Token validation failed: ${error.message}`);
      
      // Mark token as expired in our tracking system
      await this.jwtTokenActivityService.markTokenAsExpired(token);
      
      throw new UnauthorizedException('Invalid or expired token');
    }
  }

  async invalidateToken(token: string) {
    try {
      await this.jwtTokenActivityService.invalidateToken(token);
      this.logger.debug(`Token invalidated: ${token.substring(0, 10)}...`);
      return { message: 'Token invalidated successfully' };
    } catch (error) {
      this.logger.error(`Failed to invalidate token: ${error.message}`);
      throw new UnauthorizedException('Failed to invalidate token');
    }
  }
} 