"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var UsersService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsersService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const XLSX = require("xlsx");
const user_entity_1 = require("./entities/user.entity");
const email_service_1 = require("../common/services/email.service");
const user_activity_entity_1 = require("../user-activity/entities/user-activity.entity");
const permissions_service_1 = require("../permissions/permissions.service");
const login_entity_1 = require("../login/entities/login.entity");
let UsersService = UsersService_1 = class UsersService {
    usersRepository;
    activityRepository;
    loginRepository;
    emailService;
    permissionsService;
    dataSource;
    logger = new common_1.Logger(UsersService_1.name);
    constructor(usersRepository, activityRepository, loginRepository, emailService, permissionsService, dataSource) {
        this.usersRepository = usersRepository;
        this.activityRepository = activityRepository;
        this.loginRepository = loginRepository;
        this.emailService = emailService;
        this.permissionsService = permissionsService;
        this.dataSource = dataSource;
    }
    sanitizeUser(user) {
        const { emailVerified, lastLoginAt, createdAt, updatedAt, ...rest } = user;
        return rest;
    }
    async logActivity(data) {
        const activity = this.activityRepository.create(data);
        return await this.activityRepository.save(activity);
    }
    async recordActivity(params) {
        const { user, action, type, method, changedObject, ipAddress } = params;
        await this.logActivity({
            username: user.email,
            action,
            type,
            method,
            changedObject,
            ipAddress,
            user,
        });
    }
    getRoleName(roleId) {
        const roleMap = {
            '1': 'Admin',
            '2': 'Retention',
            '3': 'Reactivation',
            '4': 'Owner',
            '5': 'Escalator',
            '6': 'Agent',
            '7': 'Team Lead'
        };
        return roleMap[roleId] || 'Agent';
    }
    getRoleIdFromName(roleName) {
        const roleMap = {
            'Admin': '1',
            'Retention': '2',
            'Reactivation': '3',
            'Owner': '4',
            'Escalator': '5',
            'Agent': '6',
            'Team Lead': '7'
        };
        return roleMap[roleName] || null;
    }
    async create(createUserDto, creatorId) {
        return await this.dataSource.transaction(async (manager) => {
            const existingUser = await manager.findOne(user_entity_1.User, { where: { email: createUserDto.user.email } });
            if (existingUser)
                throw new common_1.ConflictException('Email already exists');
            const roleName = this.getRoleName(createUserDto.role_id);
            const user = manager.create(user_entity_1.User, {
                email: createUserDto.user.email,
                firstName: createUserDto.user.name.split(' ')[0] || createUserDto.user.name,
                lastName: createUserDto.user.name.split(' ').slice(1).join(' ') || '',
                role: roleName,
                isActive: true,
                status: 1,
                created_by: creatorId,
                permissions: [],
            });
            if (createUserDto.reports_to_id) {
                const reportsTo = await manager.findOne(user_entity_1.User, { where: { id: parseInt(createUserDto.reports_to_id) } });
                if (reportsTo) {
                    user.reportsTo = reportsTo;
                    this.emailService.sendReportsToNotification(reportsTo.email, `${reportsTo.firstName} ${reportsTo.lastName}`, `${user.firstName} ${user.lastName}`, user.email).catch(e => this.logger.error('Failed to send reports_to notification email:', e));
                }
            }
            const savedUser = await manager.save(user_entity_1.User, user);
            const permissionsResponse = await this.permissionsService.findAll(createUserDto.role_id);
            const allRolePermissions = [
                ...permissionsResponse.data,
                ...permissionsResponse.customer_permissions,
                ...permissionsResponse.user_permissions
            ];
            let grantedPermissions = allRolePermissions.filter(permission => createUserDto.permissions.includes(permission.id.toString()));
            if (!Array.isArray(grantedPermissions))
                grantedPermissions = [];
            savedUser.permissions = grantedPermissions;
            await manager.save(user_entity_1.User, savedUser);
            this.emailService.sendWelcomeEmail(savedUser.email).catch(e => this.logger.error('Failed to send welcome email:', e));
            this.recordActivity({
                user: savedUser,
                action: 'Create',
                type: 'User',
                method: 'POST',
                changedObject: `Created user ${savedUser.email}`,
                ipAddress: '127.0.0.1',
            }).catch(e => this.logger.error('Failed to record activity:', e));
            const userWithReportsTo = await manager.findOne(user_entity_1.User, { where: { id: savedUser.id }, relations: ['reportsTo'] });
            if (!userWithReportsTo)
                throw new common_1.InternalServerErrorException('Failed to fetch created user');
            return {
                data: {
                    id: savedUser.id,
                    user: {
                        email: savedUser.email,
                        name: `${savedUser.firstName} ${savedUser.lastName}`.trim(),
                        last_login: savedUser.lastLoginAt?.toISOString() || null,
                    },
                    granted_permissions: grantedPermissions,
                    role: {
                        id: parseInt(createUserDto.role_id),
                        name: roleName,
                        permissions: allRolePermissions,
                    },
                    reports_to: userWithReportsTo.reportsTo ? {
                        id: userWithReportsTo.reportsTo.id,
                        user: {
                            email: userWithReportsTo.reportsTo.email,
                            name: `${userWithReportsTo.reportsTo.firstName} ${userWithReportsTo.reportsTo.lastName}`,
                            last_login: userWithReportsTo.reportsTo.lastLoginAt?.toISOString() || null,
                        },
                        granted_permissions: userWithReportsTo.reportsTo.permissions || [],
                    } : null,
                    created_at: savedUser.createdAt,
                    updated_at: savedUser.updatedAt,
                    is_active: savedUser.isActive,
                },
                success: true,
                message: 'ok',
                status: 201,
            };
        });
    }
    async findAll(limit = 10, offset = 0, search = '', isActive, requestingUser) {
        try {
            const queryBuilder = this.usersRepository.createQueryBuilder('user')
                .leftJoinAndSelect('user.reportsTo', 'reportsTo');
            if (search) {
                queryBuilder.where('(LOWER(user.email) LIKE LOWER(:search) OR LOWER(user.firstName) LIKE LOWER(:search) OR LOWER(user.lastName) LIKE LOWER(:search))', { search: `%${search}%` });
            }
            if (typeof isActive === 'boolean') {
                queryBuilder.andWhere('user.isActive = :isActive', { isActive });
            }
            if (requestingUser) {
                if (requestingUser.role !== 'Admin' && requestingUser.role !== 'Owner') {
                    queryBuilder.andWhere('(user.created_by = :creatorId OR user.reports_to_id = :userId)', {
                        creatorId: requestingUser.id,
                        userId: requestingUser.id
                    });
                    queryBuilder.andWhere('user.role NOT IN (:...restrictedRoles)', { restrictedRoles: ['Admin', 'Owner'] });
                }
            }
            const totalCount = await queryBuilder.getCount();
            queryBuilder
                .skip(offset)
                .take(limit);
            const users = await queryBuilder.getMany();
            this.logger.debug(`Found ${users.length} users`);
            const userLogins = await Promise.all(users.map(async (user) => {
                const latestLogin = await this.loginRepository.findOne({
                    where: { email: user.email, status: 'success' },
                    order: { verifiedAt: 'DESC' },
                });
                return { userId: user.id, login: latestLogin };
            }));
            const formattedData = users.map(user => {
                const roleId = this.getRoleIdFromName(user.role);
                const userLogin = userLogins.find(ul => ul.userId === user.id)?.login;
                return {
                    id: user.id,
                    user: {
                        email: user.email,
                        name: `${user.firstName} ${user.lastName}`,
                        last_login: userLogin?.verifiedAt?.toISOString() || user.lastLoginAt?.toISOString() || null,
                    },
                    role: {
                        id: roleId ? parseInt(roleId) : 0,
                        name: user.role,
                        permissions: [],
                    },
                    reports_to: user.reportsTo ? {
                        id: user.reportsTo.id,
                        user: {
                            email: user.reportsTo.email,
                            name: `${user.reportsTo.firstName} ${user.reportsTo.lastName}`,
                            last_login: user.reportsTo.lastLoginAt?.toISOString() || null,
                        },
                        granted_permissions: user.reportsTo.permissions || [],
                    } : null,
                    granted_permissions: user.permissions || [],
                    is_active: user.isActive,
                };
            });
            const baseUrl = process.env.API_URL || 'http://localhost:3000';
            const next = offset + limit < totalCount
                ? `${baseUrl}/api/users/employees/?limit=${limit}&offset=${offset + limit}&search=${search}&isActive=${isActive}`
                : null;
            const previous = offset > 0
                ? `${baseUrl}/api/users/employees/?limit=${limit}&offset=${Math.max(0, offset - limit)}&search=${search}&isActive=${isActive}`
                : null;
            return {
                next,
                previous,
                count: totalCount,
                limit,
                offset,
                status: 200,
                success: true,
                message: 'List Successfully Retrieved!',
                data: formattedData,
            };
        }
        catch (error) {
            this.logger.error('Error finding users:', error);
            throw new common_1.InternalServerErrorException('Failed to fetch users');
        }
    }
    async findOne(id, requestingUser) {
        if (!id || isNaN(id))
            throw new common_1.BadRequestException('Invalid user ID');
        try {
            const user = await this.usersRepository.findOne({
                where: { id },
                relations: ['reportsTo']
            });
            if (!user)
                throw new common_1.NotFoundException(`User with ID ${id} not found`);
            if (requestingUser) {
                if (requestingUser.role !== 'Admin' && requestingUser.role !== 'Owner') {
                    if (user.created_by !== requestingUser.id && (!user.reportsTo || user.reportsTo.id !== requestingUser.id)) {
                        throw new common_1.NotFoundException(`User with ID ${id} not found`);
                    }
                    if (user.role === 'Admin' || user.role === 'Owner') {
                        throw new common_1.NotFoundException(`User with ID ${id} not found`);
                    }
                }
            }
            const roleId = this.getRoleIdFromName(user.role) || '1';
            const permissionsResponse = await this.permissionsService.findAll(roleId);
            const rolePermissions = permissionsResponse.data;
            let grantedPermissions = [];
            if (user.permissions) {
                try {
                    if (typeof user.permissions === 'string') {
                        grantedPermissions = JSON.parse(user.permissions);
                    }
                    else {
                        grantedPermissions = user.permissions;
                    }
                }
                catch (error) {
                    this.logger.error(`Error parsing permissions for user ${id}:`, error);
                    grantedPermissions = [];
                }
            }
            return {
                id: user.id,
                user: {
                    email: user.email,
                    name: `${user.firstName} ${user.lastName}`,
                    last_login: user.lastLoginAt?.toISOString() || null
                },
                granted_permissions: grantedPermissions,
                role: {
                    id: parseInt(roleId),
                    name: user.role,
                    permissions: rolePermissions
                },
                reports_to: user.reportsTo ? {
                    id: user.reportsTo.id,
                    user: {
                        email: user.reportsTo.email,
                        name: `${user.reportsTo.firstName} ${user.reportsTo.lastName}`,
                        last_login: user.reportsTo.lastLoginAt?.toISOString() || null
                    },
                    granted_permissions: user.reportsTo.permissions || []
                } : null,
                created_at: user.createdAt,
                updated_at: user.updatedAt,
                is_active: user.isActive
            };
        }
        catch (error) {
            this.logger.error(`Error finding user with ID ${id}:`, error);
            throw error instanceof common_1.NotFoundException ? error : new common_1.InternalServerErrorException();
        }
    }
    async findByEmail(email) {
        if (!email)
            throw new common_1.BadRequestException('Email is required');
        try {
            return await this.usersRepository.findOne({ where: { email } });
        }
        catch (error) {
            this.logger.error('Error finding user by email:', error);
            throw new common_1.InternalServerErrorException('Failed to fetch user by email');
        }
    }
    async update(id, updateUserDto) {
        return await this.dataSource.transaction(async (manager) => {
            const existingUser = await manager.findOne(user_entity_1.User, { where: { id }, relations: ['reportsTo'] });
            if (!existingUser) {
                throw new common_1.NotFoundException(`User with ID ${id} not found`);
            }
            if (!Array.isArray(existingUser.permissions)) {
                existingUser.permissions = [];
            }
            if (updateUserDto.user) {
                existingUser.email = updateUserDto.user.email;
                const nameParts = updateUserDto.user.name.split(' ');
                existingUser.firstName = nameParts[0] || updateUserDto.user.name;
                existingUser.lastName = nameParts.slice(1).join(' ') || '';
            }
            if (updateUserDto.role_id) {
                existingUser.role = this.getRoleName(updateUserDto.role_id);
            }
            if (updateUserDto.reports_to_id !== undefined) {
                if (updateUserDto.reports_to_id) {
                    const reportsTo = await manager.findOne(user_entity_1.User, { where: { id: parseInt(updateUserDto.reports_to_id) } });
                    if (reportsTo) {
                        existingUser.reportsTo = reportsTo;
                        this.emailService.sendReportsToNotification(reportsTo.email, `${reportsTo.firstName} ${reportsTo.lastName}`, `${existingUser.firstName} ${existingUser.lastName}`, existingUser.email).catch(e => this.logger.error('Failed to send reports_to notification email:', e));
                    }
                }
            }
            if (updateUserDto.permissions && updateUserDto.permissions.length > 0) {
                const allPermissions = await this.permissionsService.getAllPermissions();
                const grantedPermissions = allPermissions.filter(permission => updateUserDto.permissions?.includes(permission.id.toString()));
                existingUser.permissions = Array.isArray(grantedPermissions) ? grantedPermissions : [];
            }
            const updatedUser = await manager.save(user_entity_1.User, existingUser);
            this.recordActivity({
                user: updatedUser,
                action: 'Update',
                type: 'User',
                method: 'PATCH',
                changedObject: `Updated user ${updatedUser.email}`,
                ipAddress: '127.0.0.1',
            }).catch(e => this.logger.error('Failed to record activity:', e));
            const roleId = this.getRoleIdFromName(updatedUser.role) || '1';
            const permissionsResponse = await this.permissionsService.findAll(roleId);
            const rolePermissions = permissionsResponse.data;
            return {
                id: updatedUser.id,
                user: {
                    email: updatedUser.email,
                    name: `${updatedUser.firstName} ${updatedUser.lastName}`.trim(),
                    last_login: updatedUser.lastLoginAt?.toISOString() || null,
                },
                granted_permissions: updatedUser.permissions || [],
                role: {
                    id: parseInt(roleId),
                    name: updatedUser.role,
                    permissions: rolePermissions,
                },
                reports_to: updatedUser.reportsTo ? {
                    id: updatedUser.reportsTo.id,
                    user: {
                        email: updatedUser.reportsTo.email,
                        name: `${updatedUser.reportsTo.firstName} ${updatedUser.reportsTo.lastName}`.trim(),
                        last_login: updatedUser.reportsTo.lastLoginAt?.toISOString() || null,
                    },
                    granted_permissions: updatedUser.reportsTo.permissions || [],
                } : null,
                created_at: updatedUser.createdAt,
                updated_at: updatedUser.updatedAt,
                is_active: updatedUser.isActive,
            };
        });
    }
    async remove(id) {
        try {
            const user = await this.usersRepository.findOne({ where: { id } });
            if (!user) {
                throw new common_1.NotFoundException(`User with ID ${id} not found`);
            }
            user.isActive = false;
            await this.usersRepository.save(user);
            await this.recordActivity({
                user,
                action: 'Delete',
                type: 'User',
                method: 'DELETE',
                changedObject: `Deleted user ${user.email}`,
                ipAddress: '127.0.0.1',
            });
            return { status: 'success', message: `User has been successfully deactivated` };
        }
        catch (error) {
            this.logger.error(`Error removing user with ID ${id}:`, error);
            if (error instanceof common_1.NotFoundException) {
                throw error;
            }
            throw new common_1.InternalServerErrorException('Failed to remove user');
        }
    }
    async deactivate(id) {
        try {
            const user = await this.findOne(id);
            user.isActive = false;
            const updatedUser = await this.usersRepository.save(user);
            await this.recordActivity({
                user: updatedUser,
                action: 'Deactivate',
                type: 'User',
                method: 'PATCH',
                changedObject: `Deactivated user ${updatedUser.email}`,
                ipAddress: '127.0.0.1',
            });
            return { user: this.sanitizeUser(updatedUser), message: 'User is deactivated' };
        }
        catch (error) {
            this.logger.error(`Error deactivating user with ID ${id}:`, error);
            throw new common_1.InternalServerErrorException('Failed to deactivate user');
        }
    }
    async reactivate(id) {
        try {
            const user = await this.usersRepository.findOne({ where: { id } });
            if (!user) {
                throw new common_1.NotFoundException(`User with ID ${id} not found`);
            }
            user.isActive = true;
            const updatedUser = await this.usersRepository.save(user);
            await this.recordActivity({
                user: updatedUser,
                action: 'Reactivate',
                type: 'User',
                method: 'PATCH',
                changedObject: `Reactivated user ${updatedUser.email}`,
                ipAddress: '127.0.0.1',
            });
            return { user: this.sanitizeUser(updatedUser), message: 'User is reactivated' };
        }
        catch (error) {
            this.logger.error(`Error reactivating user with ID ${id}:`, error);
            throw new common_1.InternalServerErrorException('Failed to reactivate user');
        }
    }
    async findDeactivatedUsers(requestingUser, ipAddress = '127.0.0.1') {
        try {
            const deactivatedUsers = await this.usersRepository.find({ where: { isActive: false } });
            this.logger.log(`Found ${deactivatedUsers.length} deactivated users`);
            if (requestingUser) {
                await this.recordActivity({
                    user: requestingUser,
                    action: 'View Deactivated List',
                    type: 'User List',
                    method: 'GET',
                    changedObject: `Admin ${requestingUser.email} retrieved ${deactivatedUsers.length} deactivated users`,
                    ipAddress,
                });
            }
            else {
                this.logger.warn('No requesting user provided for logging deactivated users list access');
            }
            return deactivatedUsers;
        }
        catch (error) {
            this.logger.error('Error finding deactivated users:', error);
            throw new common_1.InternalServerErrorException('Failed to fetch deactivated users');
        }
    }
    async bulkCreate(file) {
        const results = { success: 0, failed: 0, errors: [] };
        try {
            const workbook = XLSX.read(file.buffer, { type: 'buffer' });
            const worksheet = workbook.Sheets[workbook.SheetNames[0]];
            const records = XLSX.utils.sheet_to_json(worksheet);
            if (records.length === 0)
                throw new common_1.BadRequestException('The Excel file is empty');
            const requiredColumns = ['email', 'firstName', 'lastName', 'role'];
            const missingColumns = requiredColumns.filter(col => !(col in records[0]));
            if (missingColumns.length > 0)
                throw new common_1.BadRequestException(`Missing columns: ${missingColumns.join(', ')}`);
            for (const record of records) {
                try {
                    const roleId = this.getRoleIdFromName(record.role?.trim() || '');
                    if (!roleId) {
                        throw new common_1.BadRequestException(`Invalid role specified: ${record.role}`);
                    }
                    const dto = {
                        permissions: [],
                        user: {
                            email: record.email?.trim() || '',
                            name: `${record.firstName?.trim() || ''} ${record.lastName?.trim() || ''}`.trim(),
                        },
                        role_id: roleId,
                    };
                    if (!dto.user.email || !dto.user.name || !dto.role_id) {
                        throw new common_1.BadRequestException('All fields (email, firstName, lastName, role) are required');
                    }
                    await this.create(dto, 1);
                    results.success++;
                }
                catch (error) {
                    results.failed++;
                    results.errors.push(`Row ${results.success + results.failed}: ${error.message}`);
                }
            }
            const message = results.failed === 0
                ? `Successfully created ${results.success} users.`
                : `Partially successful: ${results.success} created, ${results.failed} failed.`;
            return { message, details: results };
        }
        catch (error) {
            this.logger.error('Error processing Excel file:', error);
            throw new common_1.BadRequestException(error.message || 'Invalid Excel file');
        }
    }
};
exports.UsersService = UsersService;
exports.UsersService = UsersService = UsersService_1 = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(user_entity_1.User)),
    __param(1, (0, typeorm_1.InjectRepository)(user_activity_entity_1.UserActivity)),
    __param(2, (0, typeorm_1.InjectRepository)(login_entity_1.Login)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        email_service_1.EmailService,
        permissions_service_1.PermissionsService,
        typeorm_2.DataSource])
], UsersService);
//# sourceMappingURL=users.service.js.map